/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode.erasurecode;

import com.google.common.base.Preconditions;
import java.util.Collection;
import java.util.concurrent.CompletionService;
import java.util.concurrent.ExecutorCompletionService;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.SynchronousQueue;
import java.util.concurrent.ThreadFactory;
import java.util.concurrent.ThreadPoolExecutor;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.DFSUtilClient;
import org.apache.hadoop.hdfs.server.datanode.DataNode;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedBlockReconstructor;
import org.apache.hadoop.hdfs.server.datanode.erasurecode.StripedReconstructionInfo;
import org.apache.hadoop.hdfs.server.protocol.BlockECReconstructionCommand;
import org.apache.hadoop.hdfs.util.StripedBlockUtil;
import org.apache.hadoop.util.Daemon;
import org.slf4j.Logger;

@InterfaceAudience.Private
public final class ErasureCodingWorker {
    private static final Logger LOG = DataNode.LOG;
    private final DataNode datanode;
    private final Configuration conf;
    private final float xmitWeight;
    private ThreadPoolExecutor stripedReconstructionPool;
    private ThreadPoolExecutor stripedReadPool;

    public ErasureCodingWorker(Configuration conf, DataNode datanode) {
        this.datanode = datanode;
        this.conf = conf;
        this.xmitWeight = conf.getFloat("dfs.datanode.ec.reconstruction.xmits.weight", 0.5f);
        Preconditions.checkArgument((this.xmitWeight >= 0.0f ? 1 : 0) != 0, (Object)("Invalid value configured for dfs.datanode.ec.reconstruction.xmits.weight, it can not be negative value (" + this.xmitWeight + ")."));
        this.initializeStripedReadThreadPool();
        this.initializeStripedBlkReconstructionThreadPool(conf.getInt("dfs.datanode.ec.reconstruction.threads", 8));
    }

    private void initializeStripedReadThreadPool() {
        LOG.debug("Using striped reads");
        this.stripedReadPool = new ThreadPoolExecutor(0, Integer.MAX_VALUE, 60L, TimeUnit.SECONDS, new SynchronousQueue<Runnable>(), (ThreadFactory)new Daemon.DaemonFactory(){
            private final AtomicInteger threadIndex = new AtomicInteger(0);

            public Thread newThread(Runnable r) {
                Thread t = super.newThread(r);
                t.setName("stripedRead-" + this.threadIndex.getAndIncrement());
                return t;
            }
        }, new ThreadPoolExecutor.CallerRunsPolicy(){

            @Override
            public void rejectedExecution(Runnable runnable, ThreadPoolExecutor e) {
                LOG.info("Execution for striped reading rejected, Executing in current thread");
                super.rejectedExecution(runnable, e);
            }
        });
        this.stripedReadPool.allowCoreThreadTimeOut(true);
    }

    private void initializeStripedBlkReconstructionThreadPool(int numThreads) {
        LOG.debug("Using striped block reconstruction; pool threads={}", (Object)numThreads);
        this.stripedReconstructionPool = DFSUtilClient.getThreadPoolExecutor((int)2, (int)numThreads, (long)60L, new LinkedBlockingQueue(), (String)"StripedBlockReconstruction-", (boolean)false);
        this.stripedReconstructionPool.allowCoreThreadTimeOut(true);
    }

    public void processErasureCodingTasks(Collection<BlockECReconstructionCommand.BlockECReconstructionInfo> ecTasks) {
        for (BlockECReconstructionCommand.BlockECReconstructionInfo reconInfo : ecTasks) {
            int xmitsSubmitted = 0;
            try {
                StripedReconstructionInfo stripedReconInfo = new StripedReconstructionInfo(reconInfo.getExtendedBlock(), reconInfo.getErasureCodingPolicy(), reconInfo.getLiveBlockIndices(), reconInfo.getSourceDnInfos(), reconInfo.getTargetDnInfos(), reconInfo.getTargetStorageTypes(), reconInfo.getTargetStorageIDs());
                StripedBlockReconstructor task = new StripedBlockReconstructor(this, stripedReconInfo);
                if (task.hasValidTargets()) {
                    xmitsSubmitted = Math.max((int)((float)task.getXmits() * this.xmitWeight), 1);
                    this.getDatanode().incrementXmitsInProcess(xmitsSubmitted);
                    this.stripedReconstructionPool.submit(task);
                    continue;
                }
                LOG.warn("No missing internal block. Skip reconstruction for task:{}", (Object)reconInfo);
            }
            catch (Throwable e) {
                this.getDatanode().decrementXmitsInProgress(xmitsSubmitted);
                LOG.warn("Failed to reconstruct striped block {}", (Object)reconInfo.getExtendedBlock().getLocalBlock(), (Object)e);
            }
        }
    }

    DataNode getDatanode() {
        return this.datanode;
    }

    Configuration getConf() {
        return this.conf;
    }

    CompletionService<StripedBlockUtil.BlockReadStats> createReadService() {
        return new ExecutorCompletionService<StripedBlockUtil.BlockReadStats>(this.stripedReadPool);
    }

    public void shutDown() {
        this.stripedReconstructionPool.shutdown();
        this.stripedReadPool.shutdown();
    }
}

