/*
 * Decompiled with CFR 0.152.
 */
package org.apache.twill.internal;

import com.google.common.base.Function;
import com.google.common.base.Splitter;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableList;
import com.google.common.collect.ImmutableSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;
import com.google.common.collect.Sets;
import com.google.common.io.ByteStreams;
import com.google.common.io.Closeables;
import com.google.common.io.Files;
import java.io.BufferedOutputStream;
import java.io.ByteArrayOutputStream;
import java.io.Closeable;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.net.URI;
import java.net.URL;
import java.util.Collections;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.Set;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.zip.CRC32;
import java.util.zip.CheckedOutputStream;
import org.apache.twill.api.ClassAcceptor;
import org.apache.twill.filesystem.Location;
import org.apache.twill.internal.utils.Dependencies;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public final class ApplicationBundler {
    private static final Logger LOG = LoggerFactory.getLogger(ApplicationBundler.class);
    public static final String SUBDIR_CLASSES = "classes/";
    public static final String SUBDIR_LIB = "lib/";
    public static final String SUBDIR_RESOURCES = "resources/";
    private final ClassAcceptor classAcceptor;
    private final Set<String> bootstrapClassPaths;
    private final CRC32 crc32;

    public ApplicationBundler(ClassAcceptor classAcceptor) {
        this.classAcceptor = classAcceptor;
        ImmutableSet.Builder builder = ImmutableSet.builder();
        for (String classpath : Splitter.on((char)File.pathSeparatorChar).split((CharSequence)System.getProperty("sun.boot.class.path"))) {
            File file = new File(classpath);
            builder.add((Object)file.getAbsolutePath());
            try {
                builder.add((Object)file.getCanonicalPath());
            }
            catch (IOException iOException) {}
        }
        this.bootstrapClassPaths = builder.build();
        this.crc32 = new CRC32();
    }

    public ApplicationBundler(Iterable<String> excludePackages) {
        this(excludePackages, (Iterable<String>)ImmutableList.of());
    }

    public ApplicationBundler(final Iterable<String> excludePackages, final Iterable<String> includePackages) {
        this(new ClassAcceptor(){

            public boolean accept(String className, URL classUrl, URL classPathUrl) {
                for (String includePackage : includePackages) {
                    if (!className.startsWith(includePackage)) continue;
                    return true;
                }
                for (String excludePackage : excludePackages) {
                    if (!className.startsWith(excludePackage)) continue;
                    return false;
                }
                return true;
            }
        });
    }

    public void createBundle(Location target, Iterable<Class<?>> classes) throws IOException {
        this.createBundle(target, classes, (Iterable<URI>)ImmutableList.of());
    }

    public void createBundle(Location target, Class<?> clz, Class<?> ... classes) throws IOException {
        this.createBundle(target, (Iterable<Class<?>>)ImmutableSet.builder().add(clz).add((Object[])classes).build());
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void createBundle(Location target, Iterable<Class<?>> classes, Iterable<URI> resources) throws IOException {
        LOG.debug("start creating bundle {}. building a temporary file locally at first", (Object)target.getName());
        File tmpJar = File.createTempFile(target.getName(), ".tmp");
        try {
            HashSet entries = Sets.newHashSet();
            try (JarOutputStream jarOut = new JarOutputStream(new FileOutputStream(tmpJar));){
                this.findDependencies(classes, entries, jarOut);
                for (URI resource : resources) {
                    this.copyResource(resource, entries, jarOut);
                }
            }
            LOG.debug("copying temporary bundle to destination {} ({} bytes)", (Object)target, (Object)tmpJar.length());
            try {
                BufferedOutputStream os = new BufferedOutputStream(target.getOutputStream());
                try {
                    Files.copy((File)tmpJar, (OutputStream)os);
                }
                finally {
                    Closeables.closeQuietly((Closeable)os);
                }
            }
            catch (IOException e) {
                throw new IOException("failed to copy bundle from " + tmpJar.toURI() + " to " + target, e);
            }
            LOG.debug("finished creating bundle at {}", (Object)target);
        }
        finally {
            tmpJar.delete();
            LOG.debug("cleaned up local temporary for bundle {}", (Object)tmpJar.toURI());
        }
    }

    private void findDependencies(Iterable<Class<?>> classes, final Set<String> entries, final JarOutputStream jarOut) throws IOException {
        Iterable classNames = Iterables.transform(classes, (Function)new Function<Class<?>, String>(){

            public String apply(Class<?> input) {
                return input.getName();
            }
        });
        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if (classLoader == null) {
            classLoader = this.getClass().getClassLoader();
        }
        final HashSet seenClassPaths = Sets.newHashSet();
        Dependencies.findClassDependencies(classLoader, new ClassAcceptor(){

            public boolean accept(String className, URL classUrl, URL classPathUrl) {
                if (ApplicationBundler.this.bootstrapClassPaths.contains(classPathUrl.getFile())) {
                    return false;
                }
                if (!ApplicationBundler.this.classAcceptor.accept(className, classUrl, classPathUrl)) {
                    return false;
                }
                if (seenClassPaths.add(classPathUrl)) {
                    ApplicationBundler.this.putEntry(className, classUrl, classPathUrl, entries, jarOut);
                }
                return true;
            }
        }, classNames);
    }

    private void putEntry(String className, URL classUrl, URL classPathUrl, Set<String> entries, JarOutputStream jarOut) {
        String classPath = classPathUrl.getFile();
        if (classPath.endsWith(".jar")) {
            this.saveDirEntry(SUBDIR_LIB, entries, jarOut);
            this.saveEntry(SUBDIR_LIB + classPath.substring(classPath.lastIndexOf(47) + 1), classPathUrl, entries, jarOut, false);
        } else {
            this.saveDirEntry(SUBDIR_CLASSES, entries, jarOut);
            if ("file".equals(classPathUrl.getProtocol())) {
                try {
                    this.copyDir(new File(classPathUrl.toURI()), SUBDIR_CLASSES, entries, jarOut);
                }
                catch (Exception e) {
                    throw Throwables.propagate((Throwable)e);
                }
            } else {
                String entry = SUBDIR_CLASSES + className.replace('.', '/') + ".class";
                this.saveDirEntry(entry.substring(0, entry.lastIndexOf(47) + 1), entries, jarOut);
                this.saveEntry(entry, classUrl, entries, jarOut, true);
            }
        }
    }

    private void saveDirEntry(String path, Set<String> entries, JarOutputStream jarOut) {
        if (entries.contains(path)) {
            return;
        }
        try {
            String entry = "";
            for (String dir : Splitter.on((char)'/').omitEmptyStrings().split((CharSequence)path)) {
                if (!entries.add(entry = entry + dir + '/')) continue;
                JarEntry jarEntry = new JarEntry(entry);
                jarEntry.setMethod(0);
                jarEntry.setSize(0L);
                jarEntry.setCrc(0L);
                jarOut.putNextEntry(jarEntry);
                jarOut.closeEntry();
            }
        }
        catch (IOException e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private void saveEntry(String entry, URL url, Set<String> entries, JarOutputStream jarOut, boolean compress) {
        if (!entries.add(entry)) {
            return;
        }
        LOG.trace("adding bundle entry " + entry);
        try {
            JarEntry jarEntry = new JarEntry(entry);
            try (InputStream is = url.openStream();){
                if (compress) {
                    jarOut.putNextEntry(jarEntry);
                    ByteStreams.copy((InputStream)is, (OutputStream)jarOut);
                } else {
                    this.crc32.reset();
                    TransferByteOutputStream os = new TransferByteOutputStream();
                    CheckedOutputStream checkedOut = new CheckedOutputStream(os, this.crc32);
                    ByteStreams.copy((InputStream)is, (OutputStream)checkedOut);
                    checkedOut.close();
                    long size = os.size();
                    jarEntry.setMethod(0);
                    jarEntry.setSize(size);
                    jarEntry.setCrc(checkedOut.getChecksum().getValue());
                    jarOut.putNextEntry(jarEntry);
                    os.transfer(jarOut);
                }
            }
            jarOut.closeEntry();
        }
        catch (Exception e) {
            throw Throwables.propagate((Throwable)e);
        }
    }

    private void copyDir(File baseDir, String entryPrefix, Set<String> entries, JarOutputStream jarOut) throws IOException {
        LOG.trace("adding whole dir {} to bundle at '{}'", (Object)baseDir, (Object)entryPrefix);
        URI baseUri = baseDir.toURI();
        LinkedList queue = Lists.newLinkedList();
        queue.add(baseDir);
        while (!queue.isEmpty()) {
            File[] files;
            File file = (File)queue.remove();
            String entry = entryPrefix + baseUri.relativize(file.toURI()).getPath();
            if (entries.add(entry)) {
                jarOut.putNextEntry(new JarEntry(entry));
                if (file.isFile()) {
                    try {
                        Files.copy((File)file, (OutputStream)jarOut);
                    }
                    catch (IOException e) {
                        throw new IOException("failure copying from " + file.getAbsoluteFile() + " to JAR file entry " + entry, e);
                    }
                }
                jarOut.closeEntry();
            }
            if (!file.isDirectory() || (files = file.listFiles()) == null) continue;
            Collections.addAll(queue, files);
        }
    }

    private void copyResource(URI resource, Set<String> entries, JarOutputStream jarOut) throws IOException {
        File file;
        if ("file".equals(resource.getScheme()) && (file = new File(resource)).isDirectory()) {
            this.saveDirEntry(SUBDIR_RESOURCES, entries, jarOut);
            this.copyDir(file, SUBDIR_RESOURCES, entries, jarOut);
            return;
        }
        URL url = resource.toURL();
        String path = url.getFile();
        String prefix = path.endsWith(".jar") ? SUBDIR_LIB : SUBDIR_RESOURCES;
        path = prefix + path.substring(path.lastIndexOf(47) + 1);
        if (entries.add(path)) {
            this.saveDirEntry(prefix, entries, jarOut);
            jarOut.putNextEntry(new JarEntry(path));
            try (InputStream is = url.openStream();){
                ByteStreams.copy((InputStream)is, (OutputStream)jarOut);
            }
        }
    }

    private static final class TransferByteOutputStream
    extends ByteArrayOutputStream {
        private TransferByteOutputStream() {
        }

        public void transfer(OutputStream os) throws IOException {
            os.write(this.buf, 0, this.count);
        }
    }
}

