/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hive.spark.client;

import com.google.common.base.Charsets;
import com.google.common.base.Joiner;
import com.google.common.base.Preconditions;
import com.google.common.base.Strings;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import com.google.common.io.Resources;
import io.netty.channel.ChannelHandlerContext;
import io.netty.util.concurrent.Future;
import io.netty.util.concurrent.GenericFutureListener;
import io.netty.util.concurrent.Promise;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.io.OutputStreamWriter;
import java.io.Serializable;
import java.net.URI;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.Properties;
import java.util.UUID;
import java.util.concurrent.TimeoutException;
import org.apache.commons.lang3.StringUtils;
import org.apache.hadoop.hive.common.log.LogRedirector;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.shims.Utils;
import org.apache.hadoop.security.SecurityUtil;
import org.apache.hive.spark.client.BaseProtocol;
import org.apache.hive.spark.client.Job;
import org.apache.hive.spark.client.JobContext;
import org.apache.hive.spark.client.JobHandle;
import org.apache.hive.spark.client.JobHandleImpl;
import org.apache.hive.spark.client.RemoteDriver;
import org.apache.hive.spark.client.SparkClient;
import org.apache.hive.spark.client.SparkClientUtilities;
import org.apache.hive.spark.client.rpc.Rpc;
import org.apache.hive.spark.client.rpc.RpcConfiguration;
import org.apache.hive.spark.client.rpc.RpcServer;
import org.apache.spark.SparkContext;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

class SparkClientImpl
implements SparkClient {
    private static final long serialVersionUID = 1L;
    private static final Logger LOG = LoggerFactory.getLogger(SparkClientImpl.class);
    private static final long DEFAULT_SHUTDOWN_TIMEOUT = 10000L;
    private static final String OSX_TEST_OPTS = "SPARK_OSX_TEST_OPTS";
    private static final String SPARK_HOME_ENV = "SPARK_HOME";
    private static final String SPARK_HOME_KEY = "spark.home";
    private static final String DRIVER_OPTS_KEY = "spark.driver.extraJavaOptions";
    private static final String EXECUTOR_OPTS_KEY = "spark.executor.extraJavaOptions";
    private static final String DRIVER_EXTRA_CLASSPATH = "spark.driver.extraClassPath";
    private static final String EXECUTOR_EXTRA_CLASSPATH = "spark.executor.extraClassPath";
    private final Map<String, String> conf;
    private final HiveConf hiveConf;
    private final Thread driverThread;
    private final Map<String, JobHandleImpl<?>> jobs;
    private final Rpc driverRpc;
    private final ClientProtocol protocol;
    private volatile boolean isAlive;

    SparkClientImpl(RpcServer rpcServer, Map<String, String> conf, HiveConf hiveConf, String sessionid) throws IOException {
        this.conf = conf;
        this.hiveConf = hiveConf;
        this.jobs = Maps.newConcurrentMap();
        String secret = rpcServer.createSecret();
        this.driverThread = this.startDriver(rpcServer, sessionid, secret);
        this.protocol = new ClientProtocol();
        try {
            this.driverRpc = (Rpc)rpcServer.registerClient(sessionid, secret, this.protocol).get();
        }
        catch (Throwable e) {
            String errorMsg = e.getCause() instanceof TimeoutException ? "Timed out waiting for Remote Spark Driver to connect to HiveServer2.\nPossible reasons include network issues, errors in remote driver, cluster has no available resources, etc.\nPlease check YARN or Spark driver's logs for further information." : (e.getCause() instanceof InterruptedException ? "Interrupted while waiting for Remote Spark Driver to connect to HiveServer2.\nIt is possible that the query was cancelled which would cause the Spark Session to close." : "Error while waiting for Remote Spark Driver to connect back to HiveServer2.");
            LOG.error(errorMsg, e);
            this.driverThread.interrupt();
            try {
                this.driverThread.join();
            }
            catch (InterruptedException ie) {
                LOG.warn("Interrupted before driver thread was finished.", (Throwable)ie);
            }
            throw new RuntimeException(errorMsg, e);
        }
        LOG.info("Successfully connected to Remote Spark Driver at: " + this.driverRpc.getRemoteAddress());
        this.driverRpc.addListener(new Rpc.Listener(){

            @Override
            public void rpcClosed(Rpc rpc) {
                if (SparkClientImpl.this.isAlive) {
                    LOG.warn("Connection to Remote Spark Driver {} closed unexpectedly", (Object)SparkClientImpl.this.driverRpc.getRemoteAddress());
                    SparkClientImpl.this.isAlive = false;
                }
            }

            public String toString() {
                return "Connection to Remote Spark Driver Closed Unexpectedly";
            }
        });
        this.isAlive = true;
    }

    @Override
    public <T extends Serializable> JobHandle<T> submit(Job<T> job) {
        return this.protocol.submit(job, Collections.emptyList());
    }

    @Override
    public <T extends Serializable> JobHandle<T> submit(Job<T> job, List<JobHandle.Listener<T>> listeners) {
        return this.protocol.submit(job, listeners);
    }

    @Override
    public <T extends Serializable> java.util.concurrent.Future<T> run(Job<T> job) {
        return this.protocol.run(job);
    }

    @Override
    public void stop() {
        if (this.isAlive) {
            this.isAlive = false;
            try {
                this.protocol.endSession();
            }
            catch (Exception e) {
                LOG.warn("Exception while waiting for end session reply.", (Throwable)e);
            }
            finally {
                this.driverRpc.close();
            }
        }
        long endTime = System.currentTimeMillis() + 10000L;
        try {
            this.driverThread.join(10000L);
        }
        catch (InterruptedException ie) {
            LOG.debug("Interrupted before driver thread was finished.");
        }
        if (endTime - System.currentTimeMillis() <= 0L) {
            LOG.warn("Timed out shutting down remote driver, interrupting...");
            this.driverThread.interrupt();
        }
    }

    @Override
    public java.util.concurrent.Future<?> addJar(URI uri) {
        return this.run(new AddJarJob(uri.toString()));
    }

    @Override
    public java.util.concurrent.Future<?> addFile(URI uri) {
        return this.run(new AddFileJob(uri.toString()));
    }

    @Override
    public java.util.concurrent.Future<Integer> getExecutorCount() {
        return this.run(new GetExecutorCountJob());
    }

    @Override
    public java.util.concurrent.Future<Integer> getDefaultParallelism() {
        return this.run(new GetDefaultParallelismJob());
    }

    @Override
    public boolean isActive() {
        return this.isAlive && this.driverRpc.isActive();
    }

    void cancel(String jobId) {
        this.protocol.cancel(jobId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Thread startDriver(final RpcServer rpcServer, final String clientId, String secret) throws IOException {
        String regStr;
        String hiveHadoopTestClasspath;
        String sparkLogDir;
        String serverAddress = rpcServer.getAddress();
        String serverPort = String.valueOf(rpcServer.getPort());
        String sparkHome = this.conf.get(SPARK_HOME_KEY);
        if (sparkHome == null) {
            sparkHome = System.getenv(SPARK_HOME_ENV);
        }
        if (sparkHome == null) {
            sparkHome = System.getProperty(SPARK_HOME_KEY);
        }
        if ((sparkLogDir = this.conf.get("hive.spark.log.dir")) == null) {
            sparkLogDir = sparkHome == null ? "./target/" : sparkHome + "/logs/";
        }
        String osxTestOpts = "";
        if (Strings.nullToEmpty(System.getProperty("os.name")).toLowerCase().contains("mac")) {
            osxTestOpts = Strings.nullToEmpty(System.getenv(OSX_TEST_OPTS));
        }
        String driverJavaOpts = Joiner.on(" ").skipNulls().join("-Dhive.spark.log.dir=" + sparkLogDir, osxTestOpts, this.conf.get(DRIVER_OPTS_KEY));
        String executorJavaOpts = Joiner.on(" ").skipNulls().join("-Dhive.spark.log.dir=" + sparkLogDir, osxTestOpts, this.conf.get(EXECUTOR_OPTS_KEY));
        File properties = File.createTempFile("spark-submit.", ".properties");
        if (!properties.setReadable(false) || !properties.setReadable(true, true)) {
            throw new IOException("Cannot change permissions of job properties file.");
        }
        properties.deleteOnExit();
        Properties allProps = new Properties();
        try {
            URL sparkDefaultsUrl = Thread.currentThread().getContextClassLoader().getResource("spark-defaults.conf");
            if (sparkDefaultsUrl != null) {
                LOG.info("Loading spark defaults configs from: " + sparkDefaultsUrl);
                allProps.load(new ByteArrayInputStream(Resources.toByteArray(sparkDefaultsUrl)));
            }
        }
        catch (Exception e) {
            String msg = "Exception trying to load spark-defaults.conf: " + e;
            throw new IOException(msg, e);
        }
        for (Map.Entry<String, String> e : this.conf.entrySet()) {
            allProps.put(e.getKey(), this.conf.get(e.getKey()));
        }
        allProps.put("spark.client.authentication.client_id", clientId);
        allProps.put("spark.client.authentication.secret", secret);
        allProps.put(DRIVER_OPTS_KEY, driverJavaOpts);
        allProps.put(EXECUTOR_OPTS_KEY, executorJavaOpts);
        String isTesting = this.conf.get("spark.testing");
        if (isTesting != null && isTesting.equalsIgnoreCase("true") && !(hiveHadoopTestClasspath = Strings.nullToEmpty(System.getenv("HIVE_HADOOP_TEST_CLASSPATH"))).isEmpty()) {
            String extraDriverClasspath = Strings.nullToEmpty((String)allProps.get(DRIVER_EXTRA_CLASSPATH));
            if (extraDriverClasspath.isEmpty()) {
                allProps.put(DRIVER_EXTRA_CLASSPATH, hiveHadoopTestClasspath);
            } else {
                extraDriverClasspath = extraDriverClasspath.endsWith(File.pathSeparator) ? extraDriverClasspath : extraDriverClasspath + File.pathSeparator;
                allProps.put(DRIVER_EXTRA_CLASSPATH, extraDriverClasspath + hiveHadoopTestClasspath);
            }
            String extraExecutorClasspath = Strings.nullToEmpty((String)allProps.get(EXECUTOR_EXTRA_CLASSPATH));
            if (extraExecutorClasspath.isEmpty()) {
                allProps.put(EXECUTOR_EXTRA_CLASSPATH, hiveHadoopTestClasspath);
            } else {
                extraExecutorClasspath = extraExecutorClasspath.endsWith(File.pathSeparator) ? extraExecutorClasspath : extraExecutorClasspath + File.pathSeparator;
                allProps.put(EXECUTOR_EXTRA_CLASSPATH, extraExecutorClasspath + hiveHadoopTestClasspath);
            }
        }
        try (OutputStreamWriter writer = new OutputStreamWriter((OutputStream)new FileOutputStream(properties), Charsets.UTF_8);){
            allProps.store(writer, "Spark Context configuration");
        }
        String master = this.conf.get("spark.master");
        Preconditions.checkArgument(master != null, "spark.master is not defined.");
        String deployMode = this.conf.get("spark.submit.deployMode");
        LinkedList<String> argv = Lists.newLinkedList();
        if (sparkHome != null) {
            argv.add(new File(sparkHome, "bin/spark-submit").getAbsolutePath());
        } else {
            LOG.info("No spark.home provided, calling SparkSubmit directly.");
            argv.add(new File(System.getProperty("java.home"), "bin/java").getAbsolutePath());
            if (master.startsWith("local") || master.startsWith("mesos") || SparkClientUtilities.isYarnClientMode(master, deployMode) || master.startsWith("spark")) {
                String extra;
                String libPath;
                String cp;
                String mem = this.conf.get("spark.driver.memory");
                if (mem != null) {
                    argv.add("-Xms" + mem);
                    argv.add("-Xmx" + mem);
                }
                if ((cp = this.conf.get(DRIVER_EXTRA_CLASSPATH)) != null) {
                    argv.add("-classpath");
                    argv.add(cp);
                }
                if ((libPath = this.conf.get("spark.driver.extraLibPath")) != null) {
                    argv.add("-Djava.library.path=" + libPath);
                }
                if ((extra = this.conf.get(DRIVER_OPTS_KEY)) != null) {
                    for (String opt : extra.split("[ ]")) {
                        if (opt.trim().isEmpty()) continue;
                        argv.add(opt.trim());
                    }
                }
            }
            argv.add("org.apache.spark.deploy.SparkSubmit");
        }
        if (SparkClientUtilities.isYarnClusterMode(master, deployMode)) {
            String numOfExecutors;
            String executorMemory;
            String executorCores = this.conf.get("spark.executor.cores");
            if (executorCores != null) {
                argv.add("--executor-cores");
                argv.add(executorCores);
            }
            if ((executorMemory = this.conf.get("spark.executor.memory")) != null) {
                argv.add("--executor-memory");
                argv.add(executorMemory);
            }
            if ((numOfExecutors = this.conf.get("spark.executor.instances")) != null) {
                argv.add("--num-executors");
                argv.add(numOfExecutors);
            }
        }
        if ("kerberos".equals(this.hiveConf.get("hadoop.security.authentication"))) {
            String principal = SecurityUtil.getServerPrincipal((String)this.hiveConf.getVar(HiveConf.ConfVars.HIVE_SERVER2_KERBEROS_PRINCIPAL), (String)"0.0.0.0");
            String keyTabFile = this.hiveConf.getVar(HiveConf.ConfVars.HIVE_SERVER2_KERBEROS_KEYTAB);
            if (StringUtils.isNotBlank(principal) && StringUtils.isNotBlank(keyTabFile)) {
                if (this.hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_ENABLE_DOAS)) {
                    LinkedList<String> kinitArgv = Lists.newLinkedList();
                    kinitArgv.add("kinit");
                    kinitArgv.add(principal);
                    kinitArgv.add("-k");
                    kinitArgv.add("-t");
                    kinitArgv.add(keyTabFile + ";");
                    kinitArgv.addAll(argv);
                    argv = kinitArgv;
                } else {
                    argv.add("--principal");
                    argv.add(principal);
                    argv.add("--keytab");
                    argv.add(keyTabFile);
                }
            }
        }
        if (this.hiveConf.getBoolVar(HiveConf.ConfVars.HIVE_SERVER2_ENABLE_DOAS)) {
            try {
                String currentUser = Utils.getUGI().getShortUserName();
                if (!currentUser.equals(System.getProperty("user.name"))) {
                    LOG.info("Attempting impersonation of " + currentUser);
                    argv.add("--proxy-user");
                    argv.add(currentUser);
                }
            }
            catch (Exception e) {
                String msg = "Cannot obtain username: " + e;
                throw new IllegalStateException(msg, e);
            }
        }
        if ("org.apache.hive.spark.HiveKryoRegistrator".equals(regStr = this.conf.get("spark.kryo.registrator"))) {
            argv.add("--jars");
            argv.add(SparkClientUtilities.findKryoRegistratorJar(this.hiveConf));
        }
        argv.add("--properties-file");
        argv.add(properties.getAbsolutePath());
        argv.add("--class");
        argv.add(RemoteDriver.class.getName());
        String jar = "spark-internal";
        if (SparkContext.jarOfClass(this.getClass()).isDefined()) {
            jar = (String)SparkContext.jarOfClass(this.getClass()).get();
        }
        argv.add(jar);
        argv.add("--remote-host");
        argv.add(serverAddress);
        argv.add("--remote-port");
        argv.add(serverPort);
        for (String hiveSparkConfKey : RpcConfiguration.HIVE_SPARK_RSC_CONFIGS) {
            String value = RpcConfiguration.getValue(this.hiveConf, hiveSparkConfKey);
            argv.add("--remote-driver-conf");
            argv.add(String.format("%s=%s", hiveSparkConfKey, value));
        }
        String cmd = Joiner.on(" ").join(argv);
        LOG.info("Running client driver with argv: {}", (Object)cmd);
        ProcessBuilder pb = new ProcessBuilder("sh", "-c", cmd);
        pb.environment().remove("HIVE_HOME");
        pb.environment().remove("HIVE_CONF_DIR");
        String password = this.getSparkJobCredentialProviderPassword();
        if (password != null) {
            pb.environment().put("HADOOP_CREDSTORE_PASSWORD", password);
        }
        if (isTesting != null) {
            pb.environment().put("SPARK_TESTING", isTesting);
        }
        final Process child = pb.start();
        String threadName = Thread.currentThread().getName();
        final List<String> childErrorLog = Collections.synchronizedList(new ArrayList());
        LogRedirector.LogSourceCallback callback = () -> this.isAlive;
        LogRedirector.redirect("spark-submit-stdout-redir-" + threadName, new LogRedirector(child.getInputStream(), LOG, callback));
        LogRedirector.redirect("spark-submit-stderr-redir-" + threadName, new LogRedirector(child.getErrorStream(), LOG, childErrorLog, callback));
        Runnable runnable = new Runnable(){

            /*
             * WARNING - Removed try catching itself - possible behaviour change.
             */
            @Override
            public void run() {
                block8: {
                    try {
                        int exitCode = child.waitFor();
                        if (exitCode != 0) {
                            ArrayList<String> errorMessages = new ArrayList<String>();
                            List list = childErrorLog;
                            synchronized (list) {
                                for (String line : childErrorLog) {
                                    if (!StringUtils.containsIgnoreCase(line, "Error")) continue;
                                    errorMessages.add("\"" + line + "\"");
                                }
                            }
                            String errStr = errorMessages.isEmpty() ? "?" : Joiner.on(',').join(errorMessages);
                            rpcServer.cancelClient(clientId, new RuntimeException("spark-submit process failed with exit code " + exitCode + " and error " + errStr));
                            break block8;
                        }
                        LOG.info("Child process (spark-submit) exited successfully.");
                    }
                    catch (InterruptedException ie) {
                        LOG.warn("Thread waiting on the child process (spark-submit) is interrupted, killing the child process.");
                        rpcServer.cancelClient(clientId, "Thread waiting on the child porcess (spark-submit) is interrupted");
                        Thread.interrupted();
                        child.destroy();
                    }
                    catch (Exception e) {
                        String errMsg = "Exception while waiting for child process (spark-submit)";
                        LOG.warn(errMsg, (Throwable)e);
                        rpcServer.cancelClient(clientId, errMsg);
                    }
                }
            }
        };
        Thread thread = new Thread(runnable);
        thread.setDaemon(true);
        thread.setName("Driver");
        thread.start();
        return thread;
    }

    private String getSparkJobCredentialProviderPassword() {
        if (this.conf.containsKey("spark.yarn.appMasterEnv.HADOOP_CREDSTORE_PASSWORD")) {
            return this.conf.get("spark.yarn.appMasterEnv.HADOOP_CREDSTORE_PASSWORD");
        }
        if (this.conf.containsKey("spark.executorEnv.HADOOP_CREDSTORE_PASSWORD")) {
            return this.conf.get("spark.executorEnv.HADOOP_CREDSTORE_PASSWORD");
        }
        return null;
    }

    private static class GetDefaultParallelismJob
    implements Job<Integer> {
        private static final long serialVersionUID = 1L;

        private GetDefaultParallelismJob() {
        }

        @Override
        public Integer call(JobContext jc) throws Exception {
            return jc.sc().sc().defaultParallelism();
        }
    }

    private static class GetExecutorCountJob
    implements Job<Integer> {
        private static final long serialVersionUID = 1L;

        private GetExecutorCountJob() {
        }

        @Override
        public Integer call(JobContext jc) throws Exception {
            int count = jc.sc().sc().getExecutorMemoryStatus().size() - 1;
            return count;
        }
    }

    private static class AddFileJob
    implements Job<Serializable> {
        private static final long serialVersionUID = 1L;
        private final String path;

        AddFileJob() {
            this(null);
        }

        AddFileJob(String path) {
            this.path = path;
        }

        @Override
        public Serializable call(JobContext jc) throws Exception {
            jc.sc().addFile(this.path);
            return null;
        }
    }

    private static class AddJarJob
    implements Job<Serializable> {
        private static final long serialVersionUID = 1L;
        private final String path;

        AddJarJob() {
            this(null);
        }

        AddJarJob(String path) {
            this.path = path;
        }

        @Override
        public Serializable call(JobContext jc) throws Exception {
            jc.sc().addJar(this.path);
            jc.getAddedJars().put(this.path, System.currentTimeMillis());
            return null;
        }
    }

    private class ClientProtocol
    extends BaseProtocol {
        private ClientProtocol() {
        }

        <T extends Serializable> JobHandleImpl<T> submit(Job<T> job, List<JobHandle.Listener<T>> listeners) {
            final String jobId = UUID.randomUUID().toString();
            final Promise promise = SparkClientImpl.this.driverRpc.createPromise();
            final JobHandleImpl handle = new JobHandleImpl(SparkClientImpl.this, promise, jobId, listeners);
            SparkClientImpl.this.jobs.put(jobId, handle);
            final Future<Void> rpc = SparkClientImpl.this.driverRpc.call(new BaseProtocol.JobRequest<T>(jobId, job));
            LOG.debug("Send JobRequest[{}].", (Object)jobId);
            rpc.addListener((GenericFutureListener)new GenericFutureListener<Future<Void>>(){

                public void operationComplete(Future<Void> f) {
                    if (f.isSuccess()) {
                        handle.changeState(JobHandle.State.QUEUED);
                    } else if (!promise.isDone()) {
                        promise.setFailure(f.cause());
                    }
                }
            });
            promise.addListener(new GenericFutureListener<Promise<T>>(){

                public void operationComplete(Promise<T> p) {
                    if (jobId != null) {
                        SparkClientImpl.this.jobs.remove(jobId);
                    }
                    if (p.isCancelled() && !rpc.isDone()) {
                        rpc.cancel(true);
                    }
                }
            });
            return handle;
        }

        <T extends Serializable> java.util.concurrent.Future<T> run(Job<T> job) {
            Future<Serializable> rpc = SparkClientImpl.this.driverRpc.call(new BaseProtocol.SyncJobRequest<T>(job), Serializable.class);
            return rpc;
        }

        void cancel(String jobId) {
            SparkClientImpl.this.driverRpc.call(new BaseProtocol.CancelJob(jobId));
        }

        java.util.concurrent.Future<?> endSession() {
            return SparkClientImpl.this.driverRpc.call(new BaseProtocol.EndSession());
        }

        private void handle(ChannelHandlerContext ctx, BaseProtocol.Error msg) {
            LOG.warn("Error reported from Remote Spark Driver: {}", (Object)msg.cause);
        }

        private void handle(ChannelHandlerContext ctx, BaseProtocol.JobMetrics msg) {
            JobHandleImpl handle = (JobHandleImpl)SparkClientImpl.this.jobs.get(msg.jobId);
            if (handle != null) {
                handle.getMetrics().addMetrics(msg.sparkJobId, msg.stageId, msg.taskId, msg.metrics);
            } else {
                LOG.warn("Received metrics for unknown Spark job {}", (Object)msg.sparkJobId);
            }
        }

        private void handle(ChannelHandlerContext ctx, BaseProtocol.JobResult msg) {
            JobHandleImpl handle = (JobHandleImpl)SparkClientImpl.this.jobs.remove(msg.id);
            if (handle != null) {
                LOG.debug("Received result for client job {}", (Object)msg.id);
                handle.setSparkCounters(msg.sparkCounters);
                Throwable error = msg.error;
                if (error == null) {
                    handle.setSuccess(msg.result);
                } else {
                    handle.setFailure(error);
                }
            } else {
                LOG.warn("Received result for unknown client job {}", (Object)msg.id);
            }
        }

        private void handle(ChannelHandlerContext ctx, BaseProtocol.JobStarted msg) {
            JobHandleImpl handle = (JobHandleImpl)SparkClientImpl.this.jobs.get(msg.id);
            if (handle != null) {
                handle.changeState(JobHandle.State.STARTED);
            } else {
                LOG.warn("Received event for unknown client job {}", (Object)msg.id);
            }
        }

        private void handle(ChannelHandlerContext ctx, BaseProtocol.JobSubmitted msg) {
            JobHandleImpl handle = (JobHandleImpl)SparkClientImpl.this.jobs.get(msg.clientJobId);
            if (handle != null) {
                LOG.info("Received Spark job ID: {} for client job {}", (Object)msg.sparkJobId, (Object)msg.clientJobId);
                handle.addSparkJobId(msg.sparkJobId);
            } else {
                LOG.warn("Received Spark job ID: {} for unknown client job {}", (Object)msg.sparkJobId, (Object)msg.clientJobId);
            }
        }

        @Override
        protected String name() {
            return "HiveServer2 to Remote Spark Driver Connection";
        }
    }
}

