/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hive.ql.exec.spark.session;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.base.Throwables;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Maps;
import java.io.IOException;
import java.util.Collections;
import java.util.Map;
import java.util.Set;
import java.util.UUID;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.hive.common.ObjectPair;
import org.apache.hadoop.hive.conf.HiveConf;
import org.apache.hadoop.hive.ql.DriverContext;
import org.apache.hadoop.hive.ql.ErrorMsg;
import org.apache.hadoop.hive.ql.exec.spark.HiveSparkClient;
import org.apache.hadoop.hive.ql.exec.spark.HiveSparkClientFactory;
import org.apache.hadoop.hive.ql.exec.spark.session.SparkSession;
import org.apache.hadoop.hive.ql.exec.spark.status.SparkJobRef;
import org.apache.hadoop.hive.ql.metadata.HiveException;
import org.apache.hadoop.hive.ql.plan.SparkWork;
import org.apache.hadoop.hive.ql.session.SessionState;
import org.apache.spark.SparkConf;
import org.apache.spark.util.Utils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SparkSessionImpl
implements SparkSession {
    private static final Logger LOG = LoggerFactory.getLogger(SparkSession.class);
    private static final String SPARK_DIR = "_spark_session_dir";
    private static final String AM_TIMEOUT_ERR = ".*ApplicationMaster for attempt.*timed out.*";
    private static final String UNKNOWN_QUEUE_ERR = "(submitted by user.*to unknown queue:.*)\n";
    private static final String STOPPED_QUEUE_ERR = "(Queue.*is STOPPED)";
    private static final String FULL_QUEUE_ERR = "(Queue.*already has.*applications)";
    private static final String INVALILD_MEM_ERR = "(Required executor memory.*is above the max threshold.*) of this";
    private static final String INVALID_CORE_ERR = "(initial executor number.*must between min executor.*and max executor number.*)\n";
    private static Map<String, Pattern> errorPatterns;
    private volatile HiveConf conf;
    private volatile boolean isOpen;
    private final String sessionId;
    private volatile HiveSparkClient hiveSparkClient;
    private volatile Path scratchDir;
    private volatile long lastSparkJobCompletionTime;
    private final Set<String> activeJobs = Collections.newSetFromMap(new ConcurrentHashMap());
    private volatile boolean queryCompleted;
    private ReadWriteLock closeLock = new ReentrantReadWriteLock();

    public SparkSessionImpl() {
        this.sessionId = SparkSessionImpl.makeSessionId();
        SparkSessionImpl.initErrorPatterns();
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public void open(HiveConf conf) throws HiveException {
        this.closeLock.readLock().lock();
        try {
            LOG.info("Trying to open Hive on Spark session {}", (Object)this.sessionId);
            this.conf = conf;
            this.isOpen = true;
            try {
                this.hiveSparkClient = HiveSparkClientFactory.createHiveSparkClient(conf, this.sessionId);
            }
            catch (Throwable e) {
                HiveException he = this.isOpen ? this.getHiveException(e) : new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_CLOSED_SESSION, this.sessionId);
                throw he;
            }
            LOG.info("Hive on Spark session {} successfully opened", (Object)this.sessionId);
        }
        finally {
            this.closeLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public SparkJobRef submit(DriverContext driverContext, SparkWork sparkWork) throws Exception {
        this.closeLock.readLock().lock();
        try {
            Preconditions.checkState(this.isOpen, "Hive on Spark session is not open. Can't submit jobs.");
            SparkJobRef sparkJobRef = this.hiveSparkClient.execute(driverContext, sparkWork);
            return sparkJobRef;
        }
        finally {
            this.closeLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public ObjectPair<Long, Integer> getMemoryAndCores() throws Exception {
        this.closeLock.readLock().lock();
        try {
            int totalCores;
            SparkConf sparkConf = this.hiveSparkClient.getSparkConf();
            int numExecutors = this.hiveSparkClient.getExecutorCount();
            if (numExecutors <= 0) {
                ObjectPair<Long, Integer> objectPair = new ObjectPair<Long, Integer>(-1L, -1);
                return objectPair;
            }
            int executorMemoryInMB = Utils.memoryStringToMb((String)sparkConf.get("spark.executor.memory", "512m"));
            double memoryFraction = 1.0 - sparkConf.getDouble("spark.storage.memoryFraction", 0.6);
            long totalMemory = (long)((double)(numExecutors * executorMemoryInMB) * memoryFraction * 1024.0 * 1024.0);
            String masterURL = sparkConf.get("spark.master");
            if (masterURL.startsWith("spark")) {
                totalCores = sparkConf.contains("spark.default.parallelism") ? sparkConf.getInt("spark.default.parallelism", 1) : this.hiveSparkClient.getDefaultParallelism();
                totalCores = Math.max(totalCores, numExecutors);
            } else {
                int coresPerExecutor = sparkConf.getInt("spark.executor.cores", 1);
                totalCores = numExecutors * coresPerExecutor;
            }
            long memoryPerTaskInBytes = totalMemory / (long)(totalCores /= sparkConf.getInt("spark.task.cpus", 1));
            LOG.info("Hive on Spark application currently has number of executors: " + numExecutors + ", total cores: " + totalCores + ", memory per executor: " + executorMemoryInMB + " mb, memoryFraction: " + memoryFraction);
            ObjectPair<Long, Integer> objectPair = new ObjectPair<Long, Integer>(memoryPerTaskInBytes, totalCores);
            return objectPair;
        }
        finally {
            this.closeLock.readLock().unlock();
        }
    }

    @Override
    public boolean isOpen() {
        this.closeLock.readLock().lock();
        try {
            boolean bl = this.isOpen;
            return bl;
        }
        finally {
            this.closeLock.readLock().unlock();
        }
    }

    @Override
    public HiveConf getConf() {
        return this.conf;
    }

    @Override
    public String getSessionId() {
        return this.sessionId;
    }

    @Override
    public void close() {
        block7: {
            if (this.isOpen) {
                this.closeLock.writeLock().lock();
                try {
                    if (!this.isOpen) break block7;
                    LOG.info("Trying to close Hive on Spark session {}", (Object)this.sessionId);
                    this.isOpen = false;
                    if (this.hiveSparkClient != null) {
                        try {
                            this.hiveSparkClient.close();
                            LOG.info("Hive on Spark session {} successfully closed", (Object)this.sessionId);
                            this.cleanScratchDir();
                        }
                        catch (IOException e) {
                            LOG.error("Failed to close Hive on Spark session (" + this.sessionId + ")", (Throwable)e);
                        }
                    }
                    this.hiveSparkClient = null;
                    this.queryCompleted = false;
                    this.lastSparkJobCompletionTime = 0L;
                }
                finally {
                    this.closeLock.writeLock().unlock();
                }
            }
        }
    }

    private Path createScratchDir() throws IOException {
        Path parent = new Path(SessionState.get().getHdfsScratchDirURIString(), SPARK_DIR);
        Path sparkDir = new Path(parent, this.sessionId);
        FileSystem fs = sparkDir.getFileSystem((Configuration)this.conf);
        FsPermission fsPermission = new FsPermission(HiveConf.getVar(this.conf, HiveConf.ConfVars.SCRATCHDIRPERMISSION));
        fs.mkdirs(sparkDir, fsPermission);
        fs.deleteOnExit(sparkDir);
        return sparkDir;
    }

    private static void initErrorPatterns() {
        errorPatterns = Maps.newHashMap(new ImmutableMap.Builder<String, Pattern>().put(AM_TIMEOUT_ERR, Pattern.compile(AM_TIMEOUT_ERR)).put(UNKNOWN_QUEUE_ERR, Pattern.compile(UNKNOWN_QUEUE_ERR)).put(STOPPED_QUEUE_ERR, Pattern.compile(STOPPED_QUEUE_ERR)).put(FULL_QUEUE_ERR, Pattern.compile(FULL_QUEUE_ERR)).put(INVALILD_MEM_ERR, Pattern.compile(INVALILD_MEM_ERR)).put(INVALID_CORE_ERR, Pattern.compile(INVALID_CORE_ERR)).build());
    }

    @VisibleForTesting
    HiveException getHiveException(Throwable e) {
        Throwable oe = e;
        StringBuilder matchedString = new StringBuilder();
        while (e != null) {
            if (e instanceof TimeoutException) {
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_TIMEOUT, this.sessionId);
            }
            if (e instanceof InterruptedException) {
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INTERRUPTED, this.sessionId);
            }
            if (e instanceof RuntimeException) {
                String sts = Throwables.getStackTraceAsString(e);
                if (this.matches(sts, AM_TIMEOUT_ERR, matchedString)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_TIMEOUT, this.sessionId);
                }
                if (this.matches(sts, UNKNOWN_QUEUE_ERR, matchedString) || this.matches(sts, STOPPED_QUEUE_ERR, matchedString)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INVALID_QUEUE, this.sessionId, matchedString.toString());
                }
                if (this.matches(sts, FULL_QUEUE_ERR, matchedString)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_QUEUE_FULL, this.sessionId, matchedString.toString());
                }
                if (this.matches(sts, INVALILD_MEM_ERR, matchedString) || this.matches(sts, INVALID_CORE_ERR, matchedString)) {
                    return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_INVALID_RESOURCE_REQUEST, this.sessionId, matchedString.toString());
                }
                return new HiveException(e, ErrorMsg.SPARK_CREATE_CLIENT_ERROR, this.sessionId, this.getRootCause(oe));
            }
            e = e.getCause();
        }
        return new HiveException(oe, ErrorMsg.SPARK_CREATE_CLIENT_ERROR, this.sessionId, this.getRootCause(oe));
    }

    private String getRootCause(Throwable e) {
        Throwable rootCause = Throwables.getRootCause(e);
        return rootCause.getClass().getName() + ": " + rootCause.getMessage();
    }

    private boolean matches(String input, String regex, StringBuilder matchedString) {
        if (!errorPatterns.containsKey(regex)) {
            LOG.warn("No error pattern found for regex: {}", (Object)regex);
            return false;
        }
        Pattern p = errorPatterns.get(regex);
        Matcher m = p.matcher(input);
        boolean result = m.find();
        if (result && m.groupCount() == 1) {
            matchedString.append(m.group(1));
        }
        return result;
    }

    private void cleanScratchDir() throws IOException {
        if (this.scratchDir != null) {
            FileSystem fs = this.scratchDir.getFileSystem((Configuration)this.conf);
            fs.delete(this.scratchDir, true);
            this.scratchDir = null;
        }
    }

    @Override
    public Path getHDFSSessionDir() throws IOException {
        if (this.scratchDir == null) {
            this.scratchDir = this.createScratchDir();
        }
        return this.scratchDir;
    }

    @Override
    public void onQuerySubmission(String queryId) {
        this.activeJobs.add(queryId);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public boolean triggerTimeout(long sessionTimeout) {
        if (SparkSessionImpl.hasTimedOut(this.queryCompleted, this.activeJobs, this.lastSparkJobCompletionTime, sessionTimeout)) {
            this.closeLock.writeLock().lock();
            try {
                if (SparkSessionImpl.hasTimedOut(this.queryCompleted, this.activeJobs, this.lastSparkJobCompletionTime, sessionTimeout)) {
                    LOG.warn("Closing Spark session " + this.getSessionId() + " because a Spark job has not been run in the past " + sessionTimeout / 1000L + " seconds");
                    this.close();
                    boolean bl = true;
                    return bl;
                }
            }
            finally {
                this.closeLock.writeLock().unlock();
            }
        }
        return false;
    }

    private static boolean hasTimedOut(boolean queryCompleted, Set<String> activeJobs, long lastSparkJobCompletionTime, long sessionTimeout) {
        return queryCompleted && activeJobs.isEmpty() && lastSparkJobCompletionTime > 0L && System.currentTimeMillis() - lastSparkJobCompletionTime > sessionTimeout;
    }

    @Override
    public void onQueryCompletion(String queryId) {
        if (!this.queryCompleted) {
            this.queryCompleted = true;
        }
        this.activeJobs.remove(queryId);
        this.lastSparkJobCompletionTime = System.currentTimeMillis();
    }

    public static String makeSessionId() {
        return UUID.randomUUID().toString();
    }

    @VisibleForTesting
    HiveSparkClient getHiveSparkClient() {
        return this.hiveSparkClient;
    }
}

