/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.io.hfile.bucket;

import java.io.File;
import java.io.FileInputStream;
import java.io.FileNotFoundException;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.Serializable;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Comparator;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.NavigableSet;
import java.util.PriorityQueue;
import java.util.Set;
import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.ConcurrentSkipListSet;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicInteger;
import java.util.concurrent.atomic.AtomicLong;
import java.util.concurrent.atomic.LongAdder;
import java.util.concurrent.locks.Lock;
import java.util.concurrent.locks.ReentrantLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hbase.HBaseConfiguration;
import org.apache.hadoop.hbase.io.HeapSize;
import org.apache.hadoop.hbase.io.hfile.BlockCache;
import org.apache.hadoop.hbase.io.hfile.BlockCacheKey;
import org.apache.hadoop.hbase.io.hfile.BlockCacheUtil;
import org.apache.hadoop.hbase.io.hfile.BlockPriority;
import org.apache.hadoop.hbase.io.hfile.BlockType;
import org.apache.hadoop.hbase.io.hfile.CacheStats;
import org.apache.hadoop.hbase.io.hfile.Cacheable;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializer;
import org.apache.hadoop.hbase.io.hfile.CacheableDeserializerIdManager;
import org.apache.hadoop.hbase.io.hfile.CachedBlock;
import org.apache.hadoop.hbase.io.hfile.HFileBlock;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocator;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketAllocatorException;
import org.apache.hadoop.hbase.io.hfile.bucket.BucketCacheStats;
import org.apache.hadoop.hbase.io.hfile.bucket.ByteBufferIOEngine;
import org.apache.hadoop.hbase.io.hfile.bucket.CacheFullException;
import org.apache.hadoop.hbase.io.hfile.bucket.CachedEntryQueue;
import org.apache.hadoop.hbase.io.hfile.bucket.FileIOEngine;
import org.apache.hadoop.hbase.io.hfile.bucket.FileMmapEngine;
import org.apache.hadoop.hbase.io.hfile.bucket.IOEngine;
import org.apache.hadoop.hbase.io.hfile.bucket.UniqueIndexMap;
import org.apache.hadoop.hbase.io.hfile.bucket.UnsafeSharedMemoryBucketEntry;
import org.apache.hadoop.hbase.nio.ByteBuff;
import org.apache.hadoop.hbase.util.EnvironmentEdgeManager;
import org.apache.hadoop.hbase.util.HasThread;
import org.apache.hadoop.hbase.util.IdReadWriteLock;
import org.apache.hadoop.hbase.util.UnsafeAvailChecker;
import org.apache.hadoop.util.StringUtils;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.hbase.thirdparty.com.google.common.util.concurrent.ThreadFactoryBuilder;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class BucketCache
implements BlockCache,
HeapSize {
    private static final Logger LOG = LoggerFactory.getLogger(BucketCache.class);
    static final String SINGLE_FACTOR_CONFIG_NAME = "hbase.bucketcache.single.factor";
    static final String MULTI_FACTOR_CONFIG_NAME = "hbase.bucketcache.multi.factor";
    static final String MEMORY_FACTOR_CONFIG_NAME = "hbase.bucketcache.memory.factor";
    static final String EXTRA_FREE_FACTOR_CONFIG_NAME = "hbase.bucketcache.extrafreefactor";
    static final String ACCEPT_FACTOR_CONFIG_NAME = "hbase.bucketcache.acceptfactor";
    static final String MIN_FACTOR_CONFIG_NAME = "hbase.bucketcache.minfactor";
    @VisibleForTesting
    static final float DEFAULT_SINGLE_FACTOR = 0.25f;
    static final float DEFAULT_MULTI_FACTOR = 0.5f;
    static final float DEFAULT_MEMORY_FACTOR = 0.25f;
    static final float DEFAULT_MIN_FACTOR = 0.85f;
    private static final float DEFAULT_EXTRA_FREE_FACTOR = 0.1f;
    private static final float DEFAULT_ACCEPT_FACTOR = 0.95f;
    private static final int DEFAULT_FREE_ENTIRE_BLOCK_FACTOR = 2;
    private static final int statThreadPeriod = 300;
    static final int DEFAULT_WRITER_THREADS = 3;
    static final int DEFAULT_WRITER_QUEUE_ITEMS = 64;
    final IOEngine ioEngine;
    @VisibleForTesting
    final ConcurrentMap<BlockCacheKey, RAMQueueEntry> ramCache;
    @VisibleForTesting
    ConcurrentMap<BlockCacheKey, BucketEntry> backingMap;
    private volatile boolean cacheEnabled;
    @VisibleForTesting
    final ArrayList<BlockingQueue<RAMQueueEntry>> writerQueues = new ArrayList();
    @VisibleForTesting
    final WriterThread[] writerThreads;
    private volatile boolean freeInProgress = false;
    private final Lock freeSpaceLock = new ReentrantLock();
    private UniqueIndexMap<Integer> deserialiserMap = new UniqueIndexMap();
    private final LongAdder realCacheSize = new LongAdder();
    private final LongAdder heapSize = new LongAdder();
    private final LongAdder blockNumber = new LongAdder();
    private final AtomicLong accessCount = new AtomicLong();
    private static final int DEFAULT_CACHE_WAIT_TIME = 50;
    boolean wait_when_cache = false;
    private final BucketCacheStats cacheStats = new BucketCacheStats();
    private final String persistencePath;
    private final long cacheCapacity;
    private final long blockSize;
    private final int ioErrorsTolerationDuration;
    public static final int DEFAULT_ERROR_TOLERATION_DURATION = 60000;
    private volatile long ioErrorStartTime = -1L;
    @VisibleForTesting
    final IdReadWriteLock offsetLock = new IdReadWriteLock(IdReadWriteLock.ReferenceType.SOFT);
    private final NavigableSet<BlockCacheKey> blocksByHFile = new ConcurrentSkipListSet<BlockCacheKey>(new Comparator<BlockCacheKey>(){

        @Override
        public int compare(BlockCacheKey a, BlockCacheKey b) {
            int nameComparison = a.getHfileName().compareTo(b.getHfileName());
            if (nameComparison != 0) {
                return nameComparison;
            }
            if (a.getOffset() == b.getOffset()) {
                return 0;
            }
            if (a.getOffset() < b.getOffset()) {
                return -1;
            }
            return 1;
        }
    });
    private final ScheduledExecutorService scheduleThreadPool = Executors.newScheduledThreadPool(1, new ThreadFactoryBuilder().setNameFormat("BucketCacheStatsExecutor").setDaemon(true).build());
    private BucketAllocator bucketAllocator;
    private float acceptableFactor;
    private float minFactor;
    private float extraFreeFactor;
    private float singleFactor;
    private float multiFactor;
    private float memoryFactor;

    public BucketCache(String ioEngineName, long capacity, int blockSize, int[] bucketSizes, int writerThreadNum, int writerQLen, String persistencePath) throws FileNotFoundException, IOException {
        this(ioEngineName, capacity, blockSize, bucketSizes, writerThreadNum, writerQLen, persistencePath, 60000, HBaseConfiguration.create());
    }

    public BucketCache(String ioEngineName, long capacity, int blockSize, int[] bucketSizes, int writerThreadNum, int writerQLen, String persistencePath, int ioErrorsTolerationDuration, Configuration conf) throws FileNotFoundException, IOException {
        this.ioEngine = this.getIOEngineFromName(ioEngineName, capacity, persistencePath);
        this.writerThreads = new WriterThread[writerThreadNum];
        long blockNumCapacity = capacity / (long)blockSize;
        if (blockNumCapacity >= Integer.MAX_VALUE) {
            throw new IllegalArgumentException("Cache capacity is too large, only support 32TB now");
        }
        this.acceptableFactor = conf.getFloat(ACCEPT_FACTOR_CONFIG_NAME, 0.95f);
        this.minFactor = conf.getFloat(MIN_FACTOR_CONFIG_NAME, 0.85f);
        this.extraFreeFactor = conf.getFloat(EXTRA_FREE_FACTOR_CONFIG_NAME, 0.1f);
        this.singleFactor = conf.getFloat(SINGLE_FACTOR_CONFIG_NAME, 0.25f);
        this.multiFactor = conf.getFloat(MULTI_FACTOR_CONFIG_NAME, 0.5f);
        this.memoryFactor = conf.getFloat(MEMORY_FACTOR_CONFIG_NAME, 0.25f);
        this.sanityCheckConfigs();
        LOG.info("Instantiating BucketCache with acceptableFactor: " + this.acceptableFactor + ", minFactor: " + this.minFactor + ", extraFreeFactor: " + this.extraFreeFactor + ", singleFactor: " + this.singleFactor + ", multiFactor: " + this.multiFactor + ", memoryFactor: " + this.memoryFactor);
        this.cacheCapacity = capacity;
        this.persistencePath = persistencePath;
        this.blockSize = blockSize;
        this.ioErrorsTolerationDuration = ioErrorsTolerationDuration;
        this.bucketAllocator = new BucketAllocator(capacity, bucketSizes);
        for (int i = 0; i < this.writerThreads.length; ++i) {
            this.writerQueues.add(new ArrayBlockingQueue(writerQLen));
        }
        assert (this.writerQueues.size() == this.writerThreads.length);
        this.ramCache = new ConcurrentHashMap<BlockCacheKey, RAMQueueEntry>();
        this.backingMap = new ConcurrentHashMap<BlockCacheKey, BucketEntry>((int)blockNumCapacity);
        if (this.ioEngine.isPersistent() && persistencePath != null) {
            try {
                this.retrieveFromFile(bucketSizes);
            }
            catch (IOException ioex) {
                LOG.error("Can't restore from file because of", (Throwable)ioex);
            }
            catch (ClassNotFoundException cnfe) {
                LOG.error("Can't restore from file in rebuild because can't deserialise", (Throwable)cnfe);
                throw new RuntimeException(cnfe);
            }
        }
        String threadName = Thread.currentThread().getName();
        this.cacheEnabled = true;
        for (int i = 0; i < this.writerThreads.length; ++i) {
            this.writerThreads[i] = new WriterThread(this.writerQueues.get(i));
            this.writerThreads[i].setName(threadName + "-BucketCacheWriter-" + i);
            this.writerThreads[i].setDaemon(true);
        }
        this.startWriterThreads();
        this.scheduleThreadPool.scheduleAtFixedRate(new StatisticsThread(this), 300L, 300L, TimeUnit.SECONDS);
        LOG.info("Started bucket cache; ioengine=" + ioEngineName + ", capacity=" + StringUtils.byteDesc((long)capacity) + ", blockSize=" + StringUtils.byteDesc((long)blockSize) + ", writerThreadNum=" + writerThreadNum + ", writerQLen=" + writerQLen + ", persistencePath=" + persistencePath + ", bucketAllocator=" + this.bucketAllocator.getClass().getName());
    }

    private void sanityCheckConfigs() {
        Preconditions.checkArgument((this.acceptableFactor <= 1.0f && this.acceptableFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.acceptfactor must be between 0.0 and 1.0");
        Preconditions.checkArgument((this.minFactor <= 1.0f && this.minFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.minfactor must be between 0.0 and 1.0");
        Preconditions.checkArgument((this.minFactor <= this.acceptableFactor ? 1 : 0) != 0, (Object)"hbase.bucketcache.minfactor must be <= hbase.bucketcache.acceptfactor");
        Preconditions.checkArgument((this.extraFreeFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.extrafreefactor must be greater than 0.0");
        Preconditions.checkArgument((this.singleFactor <= 1.0f && this.singleFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.single.factor must be between 0.0 and 1.0");
        Preconditions.checkArgument((this.multiFactor <= 1.0f && this.multiFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.multi.factor must be between 0.0 and 1.0");
        Preconditions.checkArgument((this.memoryFactor <= 1.0f && this.memoryFactor >= 0.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.memory.factor must be between 0.0 and 1.0");
        Preconditions.checkArgument((this.singleFactor + this.multiFactor + this.memoryFactor == 1.0f ? 1 : 0) != 0, (Object)"hbase.bucketcache.single.factor, hbase.bucketcache.multi.factor, and hbase.bucketcache.memory.factor segments must add up to 1.0");
    }

    @VisibleForTesting
    protected void startWriterThreads() {
        for (WriterThread thread : this.writerThreads) {
            thread.start();
        }
    }

    @VisibleForTesting
    boolean isCacheEnabled() {
        return this.cacheEnabled;
    }

    @Override
    public long getMaxSize() {
        return this.cacheCapacity;
    }

    public String getIoEngine() {
        return this.ioEngine.toString();
    }

    private IOEngine getIOEngineFromName(String ioEngineName, long capacity, String persistencePath) throws IOException {
        if (ioEngineName.startsWith("file:") || ioEngineName.startsWith("files:")) {
            String[] filePaths = ioEngineName.substring(ioEngineName.indexOf(":") + 1).split(",");
            return new FileIOEngine(capacity, persistencePath != null, filePaths);
        }
        if (ioEngineName.startsWith("offheap")) {
            return new ByteBufferIOEngine(capacity);
        }
        if (ioEngineName.startsWith("mmap:")) {
            return new FileMmapEngine(ioEngineName.substring(5), capacity);
        }
        throw new IllegalArgumentException("Don't understand io engine name for cache- prefix with file:, files:, mmap: or offheap");
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable buf) {
        this.cacheBlock(cacheKey, buf, false);
    }

    @Override
    public void cacheBlock(BlockCacheKey cacheKey, Cacheable cachedItem, boolean inMemory) {
        this.cacheBlockWithWait(cacheKey, cachedItem, inMemory, this.wait_when_cache);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cacheBlockWithWait(BlockCacheKey cacheKey, Cacheable cachedItem, boolean inMemory, boolean wait) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("Caching key=" + cacheKey + ", item=" + cachedItem);
        }
        if (!this.cacheEnabled) {
            return;
        }
        if (this.backingMap.containsKey(cacheKey)) {
            Cacheable existingBlock = this.getBlock(cacheKey, false, false, false);
            try {
                if (BlockCacheUtil.compareCacheBlock(cachedItem, existingBlock) != 0) {
                    throw new RuntimeException("Cached block contents differ, which should not have happened.cacheKey:" + cacheKey);
                }
                String msg = "Caching an already cached block: " + cacheKey;
                msg = msg + ". This is harmless and can happen in rare cases (see HBASE-8547)";
                LOG.warn(msg);
            }
            finally {
                this.returnBlock(cacheKey, existingBlock);
            }
            return;
        }
        RAMQueueEntry re = new RAMQueueEntry(cacheKey, cachedItem, this.accessCount.incrementAndGet(), inMemory);
        if (this.ramCache.putIfAbsent(cacheKey, re) != null) {
            return;
        }
        int queueNum = (cacheKey.hashCode() & Integer.MAX_VALUE) % this.writerQueues.size();
        BlockingQueue<RAMQueueEntry> bq = this.writerQueues.get(queueNum);
        boolean successfulAddition = false;
        if (wait) {
            try {
                successfulAddition = bq.offer(re, 50L, TimeUnit.MILLISECONDS);
            }
            catch (InterruptedException e) {
                Thread.currentThread().interrupt();
            }
        } else {
            successfulAddition = bq.offer(re);
        }
        if (!successfulAddition) {
            this.ramCache.remove(cacheKey);
            this.cacheStats.failInsert();
        } else {
            this.blockNumber.increment();
            this.heapSize.add(cachedItem.heapSize());
            this.blocksByHFile.add(cacheKey);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Override
    public Cacheable getBlock(BlockCacheKey key, boolean caching, boolean repeat, boolean updateCacheMetrics) {
        if (!this.cacheEnabled) {
            return null;
        }
        RAMQueueEntry re = (RAMQueueEntry)this.ramCache.get(key);
        if (re != null) {
            if (updateCacheMetrics) {
                this.cacheStats.hit(caching, key.isPrimary(), key.getBlockType());
            }
            re.access(this.accessCount.incrementAndGet());
            return re.getData();
        }
        BucketEntry bucketEntry = (BucketEntry)this.backingMap.get(key);
        if (bucketEntry != null) {
            long start = System.nanoTime();
            ReentrantReadWriteLock lock = this.offsetLock.getLock(bucketEntry.offset());
            try {
                lock.readLock().lock();
                if (bucketEntry.equals(this.backingMap.get(key))) {
                    int len = bucketEntry.getLength();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Read offset=" + bucketEntry.offset() + ", len=" + len);
                    }
                    Cacheable cachedBlock = this.ioEngine.read(bucketEntry.offset(), len, bucketEntry.deserializerReference(this.deserialiserMap));
                    long timeTaken = System.nanoTime() - start;
                    if (updateCacheMetrics) {
                        this.cacheStats.hit(caching, key.isPrimary(), key.getBlockType());
                        this.cacheStats.ioHit(timeTaken);
                    }
                    if (cachedBlock.getMemoryType() == Cacheable.MemoryType.SHARED) {
                        bucketEntry.incrementRefCountAndGet();
                    }
                    bucketEntry.access(this.accessCount.incrementAndGet());
                    if (this.ioErrorStartTime > 0L) {
                        this.ioErrorStartTime = -1L;
                    }
                    Cacheable cacheable = cachedBlock;
                    return cacheable;
                }
            }
            catch (IOException ioex) {
                LOG.error("Failed reading block " + key + " from bucket cache", (Throwable)ioex);
                this.checkIOErrorIsTolerated();
            }
            finally {
                lock.readLock().unlock();
            }
        }
        if (!repeat && updateCacheMetrics) {
            this.cacheStats.miss(caching, key.isPrimary(), key.getBlockType());
        }
        return null;
    }

    @VisibleForTesting
    void blockEvicted(BlockCacheKey cacheKey, BucketEntry bucketEntry, boolean decrementBlockNumber) {
        this.bucketAllocator.freeBlock(bucketEntry.offset());
        this.realCacheSize.add(-1 * bucketEntry.getLength());
        this.blocksByHFile.remove(cacheKey);
        if (decrementBlockNumber) {
            this.blockNumber.decrement();
        }
    }

    @Override
    public boolean evictBlock(BlockCacheKey cacheKey) {
        return this.evictBlock(cacheKey, true);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private boolean forceEvict(BlockCacheKey cacheKey) {
        BucketEntry bucketEntry;
        block7: {
            if (!this.cacheEnabled) {
                return false;
            }
            RAMQueueEntry removedBlock = this.checkRamCache(cacheKey);
            bucketEntry = (BucketEntry)this.backingMap.get(cacheKey);
            if (bucketEntry == null) {
                if (removedBlock != null) {
                    this.cacheStats.evicted(0L, cacheKey.isPrimary());
                    return true;
                }
                return false;
            }
            ReentrantReadWriteLock lock = this.offsetLock.getLock(bucketEntry.offset());
            try {
                lock.writeLock().lock();
                if (this.backingMap.remove(cacheKey, bucketEntry)) {
                    this.blockEvicted(cacheKey, bucketEntry, removedBlock == null);
                    break block7;
                }
                boolean bl = false;
                return bl;
            }
            finally {
                lock.writeLock().unlock();
            }
        }
        this.cacheStats.evicted(bucketEntry.getCachedTime(), cacheKey.isPrimary());
        return true;
    }

    private RAMQueueEntry checkRamCache(BlockCacheKey cacheKey) {
        RAMQueueEntry removedBlock = (RAMQueueEntry)this.ramCache.remove(cacheKey);
        if (removedBlock != null) {
            this.blockNumber.decrement();
            this.heapSize.add(-1L * removedBlock.getData().heapSize());
        }
        return removedBlock;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    public boolean evictBlock(BlockCacheKey cacheKey, boolean deletedBlock) {
        if (!this.cacheEnabled) {
            return false;
        }
        RAMQueueEntry removedBlock = this.checkRamCache(cacheKey);
        BucketEntry bucketEntry = (BucketEntry)this.backingMap.get(cacheKey);
        if (bucketEntry == null) {
            if (removedBlock == null) return false;
            this.cacheStats.evicted(0L, cacheKey.isPrimary());
            return true;
        }
        ReentrantReadWriteLock lock = this.offsetLock.getLock(bucketEntry.offset());
        try {
            lock.writeLock().lock();
            int refCount = bucketEntry.getRefCount();
            if (refCount == 0) {
                if (!this.backingMap.remove(cacheKey, bucketEntry)) {
                    boolean bl = false;
                    return bl;
                }
                this.blockEvicted(cacheKey, bucketEntry, removedBlock == null);
            } else {
                if (!deletedBlock) {
                    if (LOG.isDebugEnabled()) {
                        LOG.debug("This block " + cacheKey + " is still referred by " + refCount + " readers. Can not be freed now");
                    }
                    boolean bl = false;
                    return bl;
                }
                if (LOG.isDebugEnabled()) {
                    LOG.debug("This block " + cacheKey + " is still referred by " + refCount + " readers. Can not be freed now. Hence will mark this" + " for evicting at a later point");
                }
                bucketEntry.markForEvict();
            }
        }
        finally {
            lock.writeLock().unlock();
        }
        this.cacheStats.evicted(bucketEntry.getCachedTime(), cacheKey.isPrimary());
        return true;
    }

    public void logStats() {
        long totalSize = this.bucketAllocator.getTotalSize();
        long usedSize = this.bucketAllocator.getUsedSize();
        long freeSize = totalSize - usedSize;
        long cacheSize = this.getRealCacheSize();
        LOG.info("failedBlockAdditions=" + this.cacheStats.getFailedInserts() + ", " + "totalSize=" + StringUtils.byteDesc((long)totalSize) + ", " + "freeSize=" + StringUtils.byteDesc((long)freeSize) + ", " + "usedSize=" + StringUtils.byteDesc((long)usedSize) + ", " + "cacheSize=" + StringUtils.byteDesc((long)cacheSize) + ", " + "accesses=" + this.cacheStats.getRequestCount() + ", " + "hits=" + this.cacheStats.getHitCount() + ", " + "IOhitsPerSecond=" + this.cacheStats.getIOHitsPerSecond() + ", " + "IOTimePerHit=" + String.format("%.2f", this.cacheStats.getIOTimePerHit()) + ", " + "hitRatio=" + (this.cacheStats.getHitCount() == 0L ? "0," : StringUtils.formatPercent((double)this.cacheStats.getHitRatio(), (int)2) + ", ") + "cachingAccesses=" + this.cacheStats.getRequestCachingCount() + ", " + "cachingHits=" + this.cacheStats.getHitCachingCount() + ", " + "cachingHitsRatio=" + (this.cacheStats.getHitCachingCount() == 0L ? "0," : StringUtils.formatPercent((double)this.cacheStats.getHitCachingRatio(), (int)2) + ", ") + "evictions=" + this.cacheStats.getEvictionCount() + ", " + "evicted=" + this.cacheStats.getEvictedCount() + ", " + "evictedPerRun=" + this.cacheStats.evictedPerEviction());
        this.cacheStats.reset();
    }

    public long getRealCacheSize() {
        return this.realCacheSize.sum();
    }

    private long acceptableSize() {
        return (long)Math.floor((float)this.bucketAllocator.getTotalSize() * this.acceptableFactor);
    }

    @VisibleForTesting
    long getPartitionSize(float partitionFactor) {
        return (long)Math.floor((float)this.bucketAllocator.getTotalSize() * partitionFactor * this.minFactor);
    }

    private int bucketSizesAboveThresholdCount(float minFactor) {
        BucketAllocator.IndexStatistics[] stats = this.bucketAllocator.getIndexStatistics();
        int fullCount = 0;
        for (int i = 0; i < stats.length; ++i) {
            long freeGoal = (long)Math.floor((float)stats[i].totalCount() * (1.0f - minFactor));
            freeGoal = Math.max(freeGoal, 1L);
            if (stats[i].freeCount() >= freeGoal) continue;
            ++fullCount;
        }
        return fullCount;
    }

    private void freeEntireBuckets(int completelyFreeBucketsNeeded) {
        if (completelyFreeBucketsNeeded != 0) {
            HashSet<Integer> inUseBuckets = new HashSet<Integer>();
            for (BucketEntry entry : this.backingMap.values()) {
                if (entry.getRefCount() == 0) continue;
                inUseBuckets.add(this.bucketAllocator.getBucketIndex(entry.offset()));
            }
            Set<Integer> candidateBuckets = this.bucketAllocator.getLeastFilledBuckets(inUseBuckets, completelyFreeBucketsNeeded);
            for (Map.Entry entry : this.backingMap.entrySet()) {
                if (!candidateBuckets.contains(this.bucketAllocator.getBucketIndex(((BucketEntry)entry.getValue()).offset()))) continue;
                this.evictBlock((BlockCacheKey)entry.getKey(), false);
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     * WARNING - void declaration
     */
    private void freeSpace(String why) {
        if (!this.freeSpaceLock.tryLock()) {
            return;
        }
        try {
            BucketEntryGroup bucketGroup;
            this.freeInProgress = true;
            long bytesToFreeWithoutExtra = 0L;
            StringBuilder msgBuffer = LOG.isDebugEnabled() ? new StringBuilder() : null;
            BucketAllocator.IndexStatistics[] stats = this.bucketAllocator.getIndexStatistics();
            long[] bytesToFreeForBucket = new long[stats.length];
            for (int i = 0; i < stats.length; ++i) {
                bytesToFreeForBucket[i] = 0L;
                long freeGoal = (long)Math.floor((float)stats[i].totalCount() * (1.0f - this.minFactor));
                freeGoal = Math.max(freeGoal, 1L);
                if (stats[i].freeCount() >= freeGoal) continue;
                bytesToFreeForBucket[i] = stats[i].itemSize() * (freeGoal - stats[i].freeCount());
                bytesToFreeWithoutExtra += bytesToFreeForBucket[i];
                if (msgBuffer == null) continue;
                msgBuffer.append("Free for bucketSize(" + stats[i].itemSize() + ")=" + StringUtils.byteDesc((long)bytesToFreeForBucket[i]) + ", ");
            }
            if (msgBuffer != null) {
                msgBuffer.append("Free for total=" + StringUtils.byteDesc((long)bytesToFreeWithoutExtra) + ", ");
            }
            if (bytesToFreeWithoutExtra <= 0L) {
                return;
            }
            long currentSize = this.bucketAllocator.getUsedSize();
            long totalSize = this.bucketAllocator.getTotalSize();
            if (LOG.isDebugEnabled() && msgBuffer != null) {
                LOG.debug("Free started because \"" + why + "\"; " + msgBuffer.toString() + " of current used=" + StringUtils.byteDesc((long)currentSize) + ", actual cacheSize=" + StringUtils.byteDesc((long)this.realCacheSize.sum()) + ", total=" + StringUtils.byteDesc((long)totalSize));
            }
            long bytesToFreeWithExtra = (long)Math.floor((float)bytesToFreeWithoutExtra * (1.0f + this.extraFreeFactor));
            BucketEntryGroup bucketSingle = new BucketEntryGroup(bytesToFreeWithExtra, this.blockSize, this.getPartitionSize(this.singleFactor));
            BucketEntryGroup bucketMulti = new BucketEntryGroup(bytesToFreeWithExtra, this.blockSize, this.getPartitionSize(this.multiFactor));
            BucketEntryGroup bucketMemory = new BucketEntryGroup(bytesToFreeWithExtra, this.blockSize, this.getPartitionSize(this.memoryFactor));
            for (Map.Entry<BlockCacheKey, BucketEntry> entry : this.backingMap.entrySet()) {
                switch (((BucketEntry)entry.getValue()).getPriority()) {
                    case SINGLE: {
                        bucketSingle.add(entry);
                        break;
                    }
                    case MULTI: {
                        bucketMulti.add(entry);
                        break;
                    }
                    case MEMORY: {
                        bucketMemory.add(entry);
                    }
                }
            }
            PriorityQueue<BucketEntryGroup> bucketQueue = new PriorityQueue<BucketEntryGroup>(3, Comparator.comparingLong(BucketEntryGroup::overflow));
            bucketQueue.add(bucketSingle);
            bucketQueue.add(bucketMulti);
            bucketQueue.add(bucketMemory);
            int n = 3;
            long bytesFreed = 0L;
            while ((bucketGroup = bucketQueue.poll()) != null) {
                void var17_18;
                long overflow = bucketGroup.overflow();
                if (overflow > 0L) {
                    long bucketBytesToFree = Math.min(overflow, (bytesToFreeWithoutExtra - bytesFreed) / (long)var17_18);
                    bytesFreed += bucketGroup.free(bucketBytesToFree);
                }
                --var17_18;
            }
            if (this.bucketSizesAboveThresholdCount(this.minFactor) > 0) {
                bucketQueue.clear();
                int n2 = 3;
                bucketQueue.add(bucketSingle);
                bucketQueue.add(bucketMulti);
                bucketQueue.add(bucketMemory);
                while ((bucketGroup = bucketQueue.poll()) != null) {
                    void var17_20;
                    long bucketBytesToFree = (bytesToFreeWithExtra - bytesFreed) / (long)var17_20;
                    bytesFreed += bucketGroup.free(bucketBytesToFree);
                    --var17_20;
                }
            }
            this.freeEntireBuckets(2 * this.bucketSizesAboveThresholdCount(1.0f));
            if (LOG.isDebugEnabled()) {
                long single = bucketSingle.totalSize();
                long multi = bucketMulti.totalSize();
                long memory = bucketMemory.totalSize();
                if (LOG.isDebugEnabled()) {
                    LOG.debug("Bucket cache free space completed; freed=" + StringUtils.byteDesc((long)bytesFreed) + ", " + "total=" + StringUtils.byteDesc((long)totalSize) + ", " + "single=" + StringUtils.byteDesc((long)single) + ", " + "multi=" + StringUtils.byteDesc((long)multi) + ", " + "memory=" + StringUtils.byteDesc((long)memory));
                }
            }
        }
        catch (Throwable t) {
            LOG.warn("Failed freeing space", t);
        }
        finally {
            this.cacheStats.evict();
            this.freeInProgress = false;
            this.freeSpaceLock.unlock();
        }
    }

    @VisibleForTesting
    static List<RAMQueueEntry> getRAMQueueEntries(BlockingQueue<RAMQueueEntry> q, List<RAMQueueEntry> receptacle) throws InterruptedException {
        receptacle.clear();
        receptacle.add(q.take());
        q.drainTo(receptacle);
        return receptacle;
    }

    private void persistToFile() throws IOException {
        assert (!this.cacheEnabled);
        FileOutputStream fos = null;
        ObjectOutputStream oos = null;
        try {
            if (!this.ioEngine.isPersistent()) {
                throw new IOException("Attempt to persist non-persistent cache mappings!");
            }
            fos = new FileOutputStream(this.persistencePath, false);
            oos = new ObjectOutputStream(fos);
            oos.writeLong(this.cacheCapacity);
            oos.writeUTF(this.ioEngine.getClass().getName());
            oos.writeUTF(this.backingMap.getClass().getName());
            oos.writeObject(this.deserialiserMap);
            oos.writeObject(this.backingMap);
        }
        finally {
            if (oos != null) {
                oos.close();
            }
            if (fos != null) {
                fos.close();
            }
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void retrieveFromFile(int[] bucketSizes) throws IOException, BucketAllocatorException, ClassNotFoundException {
        File persistenceFile = new File(this.persistencePath);
        if (!persistenceFile.exists()) {
            return;
        }
        assert (!this.cacheEnabled);
        FileInputStream fis = null;
        ObjectInputStream ois = null;
        try {
            BucketAllocator allocator;
            if (!this.ioEngine.isPersistent()) {
                throw new IOException("Attempt to restore non-persistent cache mappings!");
            }
            fis = new FileInputStream(this.persistencePath);
            ois = new ObjectInputStream(fis);
            long capacitySize = ois.readLong();
            if (capacitySize != this.cacheCapacity) {
                throw new IOException("Mismatched cache capacity:" + StringUtils.byteDesc((long)capacitySize) + ", expected: " + StringUtils.byteDesc((long)this.cacheCapacity));
            }
            String ioclass = ois.readUTF();
            String mapclass = ois.readUTF();
            if (!this.ioEngine.getClass().getName().equals(ioclass)) {
                throw new IOException("Class name for IO engine mismatch: " + ioclass + ", expected:" + this.ioEngine.getClass().getName());
            }
            if (!this.backingMap.getClass().getName().equals(mapclass)) {
                throw new IOException("Class name for cache map mismatch: " + mapclass + ", expected:" + this.backingMap.getClass().getName());
            }
            UniqueIndexMap deserMap = (UniqueIndexMap)ois.readObject();
            ConcurrentHashMap backingMapFromFile = (ConcurrentHashMap)ois.readObject();
            this.bucketAllocator = allocator = new BucketAllocator(this.cacheCapacity, bucketSizes, backingMapFromFile, this.realCacheSize);
            this.deserialiserMap = deserMap;
            this.backingMap = backingMapFromFile;
        }
        finally {
            if (ois != null) {
                ois.close();
            }
            if (fis != null) {
                fis.close();
            }
            if (!persistenceFile.delete()) {
                throw new IOException("Failed deleting persistence file " + persistenceFile.getAbsolutePath());
            }
        }
    }

    private void checkIOErrorIsTolerated() {
        long now = EnvironmentEdgeManager.currentTime();
        if (this.ioErrorStartTime > 0L) {
            if (this.cacheEnabled && now - this.ioErrorStartTime > (long)this.ioErrorsTolerationDuration) {
                LOG.error("IO errors duration time has exceeded " + this.ioErrorsTolerationDuration + "ms, disabling cache, please check your IOEngine");
                this.disableCache();
            }
        } else {
            this.ioErrorStartTime = now;
        }
    }

    private void disableCache() {
        if (!this.cacheEnabled) {
            return;
        }
        this.cacheEnabled = false;
        this.ioEngine.shutdown();
        this.scheduleThreadPool.shutdown();
        for (int i = 0; i < this.writerThreads.length; ++i) {
            this.writerThreads[i].interrupt();
        }
        this.ramCache.clear();
        if (!this.ioEngine.isPersistent() || this.persistencePath == null) {
            this.backingMap.clear();
        }
    }

    private void join() throws InterruptedException {
        for (int i = 0; i < this.writerThreads.length; ++i) {
            this.writerThreads[i].join();
        }
    }

    @Override
    public void shutdown() {
        this.disableCache();
        LOG.info("Shutdown bucket cache: IO persistent=" + this.ioEngine.isPersistent() + "; path to write=" + this.persistencePath);
        if (this.ioEngine.isPersistent() && this.persistencePath != null) {
            try {
                this.join();
                this.persistToFile();
            }
            catch (IOException ex) {
                LOG.error("Unable to persist data on exit: " + ex.toString(), (Throwable)ex);
            }
            catch (InterruptedException e) {
                LOG.warn("Failed to persist data on exit", (Throwable)e);
            }
        }
    }

    @Override
    public CacheStats getStats() {
        return this.cacheStats;
    }

    public BucketAllocator getAllocator() {
        return this.bucketAllocator;
    }

    public long heapSize() {
        return this.heapSize.sum();
    }

    @Override
    public long size() {
        return this.realCacheSize.sum();
    }

    @Override
    public long getCurrentDataSize() {
        return this.size();
    }

    @Override
    public long getFreeSize() {
        return this.bucketAllocator.getFreeSize();
    }

    @Override
    public long getBlockCount() {
        return this.blockNumber.sum();
    }

    @Override
    public long getDataBlockCount() {
        return this.getBlockCount();
    }

    @Override
    public long getCurrentSize() {
        return this.bucketAllocator.getUsedSize();
    }

    @Override
    public int evictBlocksByHfileName(String hfileName) {
        NavigableSet<BlockCacheKey> keySet = this.blocksByHFile.subSet(new BlockCacheKey(hfileName, Long.MIN_VALUE), true, new BlockCacheKey(hfileName, Long.MAX_VALUE), true);
        int numEvicted = 0;
        for (BlockCacheKey key : keySet) {
            if (!this.evictBlock(key)) continue;
            ++numEvicted;
        }
        return numEvicted;
    }

    void stopWriterThreads() throws InterruptedException {
        for (WriterThread writerThread : this.writerThreads) {
            writerThread.disableWriter();
            writerThread.interrupt();
            writerThread.join();
        }
    }

    @Override
    public Iterator<CachedBlock> iterator() {
        final Iterator i = this.backingMap.entrySet().iterator();
        return new Iterator<CachedBlock>(){
            private final long now = System.nanoTime();

            @Override
            public boolean hasNext() {
                return i.hasNext();
            }

            @Override
            public CachedBlock next() {
                final Map.Entry e = (Map.Entry)i.next();
                return new CachedBlock(){

                    public String toString() {
                        return BlockCacheUtil.toString(this, now);
                    }

                    @Override
                    public BlockPriority getBlockPriority() {
                        return ((BucketEntry)e.getValue()).getPriority();
                    }

                    @Override
                    public BlockType getBlockType() {
                        return null;
                    }

                    @Override
                    public long getOffset() {
                        return ((BlockCacheKey)e.getKey()).getOffset();
                    }

                    @Override
                    public long getSize() {
                        return ((BucketEntry)e.getValue()).getLength();
                    }

                    @Override
                    public long getCachedTime() {
                        return ((BucketEntry)e.getValue()).getCachedTime();
                    }

                    @Override
                    public String getFilename() {
                        return ((BlockCacheKey)e.getKey()).getHfileName();
                    }

                    @Override
                    public int compareTo(CachedBlock other) {
                        int diff = this.getFilename().compareTo(other.getFilename());
                        if (diff != 0) {
                            return diff;
                        }
                        diff = Long.compare(this.getOffset(), other.getOffset());
                        if (diff != 0) {
                            return diff;
                        }
                        if (other.getCachedTime() < 0L || this.getCachedTime() < 0L) {
                            throw new IllegalStateException("" + this.getCachedTime() + ", " + other.getCachedTime());
                        }
                        return Long.compare(other.getCachedTime(), this.getCachedTime());
                    }

                    public int hashCode() {
                        return ((BlockCacheKey)e.getKey()).hashCode();
                    }

                    public boolean equals(Object obj) {
                        if (obj instanceof CachedBlock) {
                            CachedBlock cb = (CachedBlock)obj;
                            return this.compareTo(cb) == 0;
                        }
                        return false;
                    }
                };
            }

            @Override
            public void remove() {
                throw new UnsupportedOperationException();
            }
        };
    }

    @Override
    public BlockCache[] getBlockCaches() {
        return null;
    }

    @Override
    public void returnBlock(BlockCacheKey cacheKey, Cacheable block) {
        int refCount;
        BucketEntry bucketEntry;
        if (block.getMemoryType() == Cacheable.MemoryType.SHARED && (bucketEntry = (BucketEntry)this.backingMap.get(cacheKey)) != null && (refCount = bucketEntry.decrementRefCountAndGet()) == 0 && bucketEntry.isMarkedForEvict()) {
            this.forceEvict(cacheKey);
        }
    }

    @VisibleForTesting
    public int getRefCount(BlockCacheKey cacheKey) {
        BucketEntry bucketEntry = (BucketEntry)this.backingMap.get(cacheKey);
        if (bucketEntry != null) {
            return bucketEntry.getRefCount();
        }
        return 0;
    }

    float getAcceptableFactor() {
        return this.acceptableFactor;
    }

    float getMinFactor() {
        return this.minFactor;
    }

    float getExtraFreeFactor() {
        return this.extraFreeFactor;
    }

    float getSingleFactor() {
        return this.singleFactor;
    }

    float getMultiFactor() {
        return this.multiFactor;
    }

    float getMemoryFactor() {
        return this.memoryFactor;
    }

    @VisibleForTesting
    static class RAMQueueEntry {
        private BlockCacheKey key;
        private Cacheable data;
        private long accessCounter;
        private boolean inMemory;

        public RAMQueueEntry(BlockCacheKey bck, Cacheable data, long accessCounter, boolean inMemory) {
            this.key = bck;
            this.data = data;
            this.accessCounter = accessCounter;
            this.inMemory = inMemory;
        }

        public Cacheable getData() {
            return this.data;
        }

        public BlockCacheKey getKey() {
            return this.key;
        }

        public void access(long accessCounter) {
            this.accessCounter = accessCounter;
        }

        public BucketEntry writeToCache(IOEngine ioEngine, BucketAllocator bucketAllocator, UniqueIndexMap<Integer> deserialiserMap, LongAdder realCacheSize) throws CacheFullException, IOException, BucketAllocatorException {
            int len = this.data.getSerializedLength();
            if (len == 0) {
                return null;
            }
            long offset = bucketAllocator.allocateBlock(len);
            BucketEntry bucketEntry = ioEngine.usesSharedMemory() ? (UnsafeAvailChecker.isAvailable() ? new UnsafeSharedMemoryBucketEntry(offset, len, this.accessCounter, this.inMemory) : new SharedMemoryBucketEntry(offset, len, this.accessCounter, this.inMemory)) : new BucketEntry(offset, len, this.accessCounter, this.inMemory);
            bucketEntry.setDeserialiserReference(this.data.getDeserializer(), deserialiserMap);
            try {
                if (this.data instanceof HFileBlock) {
                    HFileBlock block = (HFileBlock)this.data;
                    ByteBuff sliceBuf = block.getBufferReadOnly();
                    ByteBuffer metadata = block.getMetaData();
                    if (LOG.isTraceEnabled()) {
                        LOG.trace("Write offset=" + offset + ", len=" + len);
                    }
                    ioEngine.write(sliceBuf, offset);
                    ioEngine.write(metadata, offset + (long)len - (long)metadata.limit());
                } else {
                    ByteBuffer bb = ByteBuffer.allocate(len);
                    this.data.serialize(bb);
                    ioEngine.write(bb, offset);
                }
            }
            catch (IOException ioe) {
                bucketAllocator.freeBlock(offset);
                throw ioe;
            }
            realCacheSize.add(len);
            return bucketEntry;
        }
    }

    private class BucketEntryGroup {
        private CachedEntryQueue queue;
        private long totalSize = 0L;
        private long bucketSize;

        public BucketEntryGroup(long bytesToFree, long blockSize, long bucketSize) {
            this.bucketSize = bucketSize;
            this.queue = new CachedEntryQueue(bytesToFree, blockSize);
            this.totalSize = 0L;
        }

        public void add(Map.Entry<BlockCacheKey, BucketEntry> block) {
            this.totalSize += (long)block.getValue().getLength();
            this.queue.add(block);
        }

        public long free(long toFree) {
            Map.Entry<BlockCacheKey, BucketEntry> entry;
            long freedBytes = 0L;
            while ((entry = this.queue.pollLast()) != null) {
                if (BucketCache.this.evictBlock(entry.getKey(), false)) {
                    freedBytes += (long)entry.getValue().getLength();
                }
                if (freedBytes < toFree) continue;
                return freedBytes;
            }
            return freedBytes;
        }

        public long overflow() {
            return this.totalSize - this.bucketSize;
        }

        public long totalSize() {
            return this.totalSize;
        }
    }

    static class SharedMemoryBucketEntry
    extends BucketEntry {
        private static final long serialVersionUID = -2187147283772338481L;
        private volatile boolean markedForEvict;
        private AtomicInteger refCount = new AtomicInteger(0);

        SharedMemoryBucketEntry(long offset, int length, long accessCounter, boolean inMemory) {
            super(offset, length, accessCounter, inMemory);
        }

        @Override
        protected int getRefCount() {
            return this.refCount.get();
        }

        @Override
        protected int incrementRefCountAndGet() {
            return this.refCount.incrementAndGet();
        }

        @Override
        protected int decrementRefCountAndGet() {
            return this.refCount.decrementAndGet();
        }

        @Override
        protected boolean isMarkedForEvict() {
            return this.markedForEvict;
        }

        @Override
        protected void markForEvict() {
            this.markedForEvict = true;
        }
    }

    static class BucketEntry
    implements Serializable {
        private static final long serialVersionUID = -6741504807982257534L;
        static final Comparator<BucketEntry> COMPARATOR = new Comparator<BucketEntry>(){

            @Override
            public int compare(BucketEntry o1, BucketEntry o2) {
                return Long.compare(o2.accessCounter, o1.accessCounter);
            }
        };
        private int offsetBase;
        private int length;
        private byte offset1;
        byte deserialiserIndex;
        private volatile long accessCounter;
        private BlockPriority priority;
        private final long cachedTime = System.nanoTime();

        BucketEntry(long offset, int length, long accessCounter, boolean inMemory) {
            this.setOffset(offset);
            this.length = length;
            this.accessCounter = accessCounter;
            this.priority = inMemory ? BlockPriority.MEMORY : BlockPriority.SINGLE;
        }

        long offset() {
            long o = (long)this.offsetBase & 0xFFFFFFFFL;
            return (o += ((long)this.offset1 & 0xFFL) << 32) << 8;
        }

        private void setOffset(long value) {
            assert ((value & 0xFFL) == 0L);
            this.offsetBase = (int)(value >>= 8);
            this.offset1 = (byte)(value >> 32);
        }

        public int getLength() {
            return this.length;
        }

        protected CacheableDeserializer<Cacheable> deserializerReference(UniqueIndexMap<Integer> deserialiserMap) {
            return CacheableDeserializerIdManager.getDeserializer(deserialiserMap.unmap(this.deserialiserIndex));
        }

        protected void setDeserialiserReference(CacheableDeserializer<Cacheable> deserializer, UniqueIndexMap<Integer> deserialiserMap) {
            this.deserialiserIndex = (byte)deserialiserMap.map(deserializer.getDeserialiserIdentifier());
        }

        public void access(long accessCounter) {
            this.accessCounter = accessCounter;
            if (this.priority == BlockPriority.SINGLE) {
                this.priority = BlockPriority.MULTI;
            }
        }

        public BlockPriority getPriority() {
            return this.priority;
        }

        public long getCachedTime() {
            return this.cachedTime;
        }

        protected int getRefCount() {
            return 0;
        }

        protected int incrementRefCountAndGet() {
            return 0;
        }

        protected int decrementRefCountAndGet() {
            return 0;
        }

        protected boolean isMarkedForEvict() {
            return false;
        }

        protected void markForEvict() {
        }
    }

    @VisibleForTesting
    class WriterThread
    extends HasThread {
        private final BlockingQueue<RAMQueueEntry> inputQueue;
        private volatile boolean writerEnabled;

        WriterThread(BlockingQueue<RAMQueueEntry> queue) {
            super("BucketCacheWriterThread");
            this.writerEnabled = true;
            this.inputQueue = queue;
        }

        @VisibleForTesting
        void disableWriter() {
            this.writerEnabled = false;
        }

        public void run() {
            List<RAMQueueEntry> entries = new ArrayList<RAMQueueEntry>();
            try {
                while (BucketCache.this.cacheEnabled && this.writerEnabled) {
                    try {
                        try {
                            entries = BucketCache.getRAMQueueEntries(this.inputQueue, entries);
                        }
                        catch (InterruptedException ie) {
                            if (!BucketCache.this.cacheEnabled) break;
                        }
                        this.doDrain(entries);
                    }
                    catch (Exception ioe) {
                        LOG.error("WriterThread encountered error", (Throwable)ioe);
                    }
                }
            }
            catch (Throwable t) {
                LOG.warn("Failed doing drain", t);
            }
            LOG.info(this.getName() + " exiting, cacheEnabled=" + BucketCache.this.cacheEnabled);
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @VisibleForTesting
        void doDrain(List<RAMQueueEntry> entries) throws InterruptedException {
            if (entries.isEmpty()) {
                return;
            }
            int size = entries.size();
            BucketEntry[] bucketEntries = new BucketEntry[size];
            int index = 0;
            while (BucketCache.this.cacheEnabled && index < size) {
                RAMQueueEntry re = null;
                try {
                    BucketEntry bucketEntry;
                    re = entries.get(index);
                    if (re == null) {
                        LOG.warn("Couldn't get entry or changed on us; who else is messing with it?");
                        ++index;
                        continue;
                    }
                    bucketEntries[index] = bucketEntry = re.writeToCache(BucketCache.this.ioEngine, BucketCache.this.bucketAllocator, BucketCache.this.deserialiserMap, BucketCache.this.realCacheSize);
                    if (BucketCache.this.ioErrorStartTime > 0L) {
                        BucketCache.this.ioErrorStartTime = -1L;
                    }
                    ++index;
                }
                catch (BucketAllocatorException fle) {
                    LOG.warn("Failed allocation for " + (re == null ? "" : re.getKey()) + "; " + fle);
                    bucketEntries[index] = null;
                    ++index;
                }
                catch (CacheFullException cfe) {
                    if (!BucketCache.this.freeInProgress) {
                        BucketCache.this.freeSpace("Full!");
                        continue;
                    }
                    Thread.sleep(50L);
                }
                catch (IOException ioex) {
                    LOG.error("Failed writing to bucket cache", (Throwable)ioex);
                    BucketCache.this.checkIOErrorIsTolerated();
                }
            }
            try {
                BucketCache.this.ioEngine.sync();
            }
            catch (IOException ioex) {
                LOG.error("Failed syncing IO engine", (Throwable)ioex);
                BucketCache.this.checkIOErrorIsTolerated();
                for (int i = 0; i < entries.size(); ++i) {
                    if (bucketEntries[i] == null) continue;
                    BucketCache.this.bucketAllocator.freeBlock(bucketEntries[i].offset());
                    bucketEntries[i] = null;
                }
            }
            for (int i = 0; i < size; ++i) {
                RAMQueueEntry ramCacheEntry;
                BlockCacheKey key = entries.get(i).getKey();
                if (bucketEntries[i] != null) {
                    BucketCache.this.backingMap.put(key, bucketEntries[i]);
                }
                if ((ramCacheEntry = (RAMQueueEntry)BucketCache.this.ramCache.remove(key)) != null) {
                    BucketCache.this.heapSize.add(-1L * entries.get(i).getData().heapSize());
                    continue;
                }
                if (bucketEntries[i] == null) continue;
                ReentrantReadWriteLock lock = BucketCache.this.offsetLock.getLock(bucketEntries[i].offset());
                try {
                    lock.writeLock().lock();
                    if (!BucketCache.this.backingMap.remove(key, bucketEntries[i])) continue;
                    BucketCache.this.blockEvicted(key, bucketEntries[i], false);
                    continue;
                }
                finally {
                    lock.writeLock().unlock();
                }
            }
            long used = BucketCache.this.bucketAllocator.getUsedSize();
            if (used > BucketCache.this.acceptableSize()) {
                BucketCache.this.freeSpace("Used=" + used + " > acceptable=" + BucketCache.this.acceptableSize());
            }
        }
    }

    private static class StatisticsThread
    extends Thread {
        private final BucketCache bucketCache;

        public StatisticsThread(BucketCache bucketCache) {
            super("BucketCacheStatsThread");
            this.setDaemon(true);
            this.bucketCache = bucketCache;
        }

        @Override
        public void run() {
            this.bucketCache.logStats();
        }
    }
}

