/*
 * Decompiled with CFR 0.152.
 */
package org.apache.slider.common.tools;

import com.google.common.base.Preconditions;
import java.io.Closeable;
import java.io.File;
import java.io.FileNotFoundException;
import java.io.FilenameFilter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Map;
import java.util.zip.ZipEntry;
import java.util.zip.ZipFile;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FSDataOutputStream;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.fs.permission.FsPermission;
import org.apache.hadoop.io.IOUtils;
import org.apache.hadoop.yarn.api.records.LocalResource;
import org.apache.hadoop.yarn.api.records.LocalResourceType;
import org.apache.hadoop.yarn.api.records.LocalResourceVisibility;
import org.apache.hadoop.yarn.util.ConverterUtils;
import org.apache.hadoop.yarn.util.Records;
import org.apache.slider.common.tools.SliderUtils;
import org.apache.slider.core.exceptions.BadClusterStateException;
import org.apache.slider.core.exceptions.SliderException;
import org.apache.slider.core.exceptions.UnknownApplicationInstanceException;
import org.apache.slider.core.persist.InstancePaths;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class CoreFileSystem {
    private static final Logger log = LoggerFactory.getLogger(CoreFileSystem.class);
    protected final FileSystem fileSystem;
    protected final Configuration configuration;

    public CoreFileSystem(FileSystem fileSystem, Configuration configuration) {
        Preconditions.checkNotNull((Object)fileSystem, (Object)"Cannot create a CoreFileSystem with a null FileSystem");
        Preconditions.checkNotNull((Object)configuration, (Object)"Cannot create a CoreFileSystem with a null Configuration");
        this.fileSystem = fileSystem;
        this.configuration = configuration;
    }

    public CoreFileSystem(Configuration configuration) throws IOException {
        Preconditions.checkNotNull((Object)configuration, (Object)"Cannot create a CoreFileSystem with a null Configuration");
        this.fileSystem = FileSystem.get((Configuration)configuration);
        this.configuration = this.fileSystem.getConf();
    }

    public Path getTempPathForCluster(String clustername) {
        Path clusterDir = this.buildClusterDirPath(clustername);
        return new Path(clusterDir, "tmp");
    }

    public FileSystem getFileSystem() {
        return this.fileSystem;
    }

    public String toString() {
        StringBuilder sb = new StringBuilder("CoreFileSystem{");
        sb.append("fileSystem=").append(this.fileSystem.getUri());
        sb.append('}');
        return sb.toString();
    }

    public Path buildClusterDirPath(String clustername) {
        Preconditions.checkNotNull((Object)clustername);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "cluster/" + clustername);
    }

    public Path buildAppDefDirPath(String clustername) {
        Path path = this.buildClusterDirPath(clustername);
        return new Path(path, "appdef");
    }

    public Path buildAddonDirPath(String clustername, String addonId) {
        Preconditions.checkNotNull((Object)addonId);
        Path path = this.buildClusterDirPath(clustername);
        return new Path(path, "addons/" + addonId);
    }

    public Path buildPackageDirPath(String packageName, String packageVersion) {
        Preconditions.checkNotNull((Object)packageName);
        Path path = this.getBaseApplicationPath();
        path = new Path(path, "package/" + packageName);
        if (SliderUtils.isSet(packageVersion)) {
            path = new Path(path, packageVersion);
        }
        return path;
    }

    public Path buildClusterSecurityDirPath(String clusterName) {
        Preconditions.checkNotNull((Object)clusterName);
        Path path = this.buildClusterDirPath(clusterName);
        return new Path(path, "security");
    }

    public Path buildKeytabInstallationDirPath(String keytabFolder) {
        Preconditions.checkNotNull((Object)keytabFolder);
        Path path = this.getBaseApplicationPath();
        return new Path(path, "keytabs/" + keytabFolder);
    }

    public Path buildKeytabPath(String keytabDir, String keytabName, String clusterName) {
        Path homePath = this.getHomeDirectory();
        Path baseKeytabDir = keytabDir != null ? new Path(homePath, keytabDir) : new Path(this.buildClusterDirPath(clusterName), "keytabs");
        return keytabName == null ? baseKeytabDir : new Path(baseKeytabDir, keytabName);
    }

    public Path createClusterDirectories(String clustername, Configuration conf) throws IOException, SliderException {
        Path clusterDirectory = this.buildClusterDirPath(clustername);
        InstancePaths instancePaths = new InstancePaths(clusterDirectory);
        this.createClusterDirectories(instancePaths);
        return clusterDirectory;
    }

    public void createClusterDirectories(InstancePaths instancePaths) throws IOException, SliderException {
        Path instanceDir = instancePaths.instanceDir;
        this.verifyDirectoryNonexistent(instanceDir);
        FsPermission clusterPerms = this.getInstanceDirectoryPermissions();
        this.createWithPermissions(instanceDir, clusterPerms);
        this.createWithPermissions(instancePaths.snapshotConfPath, clusterPerms);
        this.createWithPermissions(instancePaths.generatedConfPath, clusterPerms);
        this.createWithPermissions(instancePaths.historyPath, clusterPerms);
        this.createWithPermissions(instancePaths.tmpPathAM, clusterPerms);
        String dataOpts = this.configuration.get("slider.data.directory.permissions", "750");
        log.debug("Setting data directory permissions to {}", (Object)dataOpts);
        this.createWithPermissions(instancePaths.dataPath, new FsPermission(dataOpts));
    }

    public void createWithPermissions(Path dir, FsPermission clusterPerms) throws IOException, BadClusterStateException {
        if (this.fileSystem.isFile(dir)) {
            throw new BadClusterStateException("Cannot create a directory over a file %s", dir);
        }
        log.debug("mkdir {} with perms {}", (Object)dir, (Object)clusterPerms);
        this.fileSystem.getConf().set("fs.permissions.umask-mode", "000");
        this.fileSystem.mkdirs(dir, clusterPerms);
        this.fileSystem.setPermission(dir, clusterPerms);
    }

    public FsPermission getPathPermissions(Path path) throws IOException {
        FileStatus status = this.fileSystem.getFileStatus(path);
        return status.getPermission();
    }

    public FsPermission getInstanceDirectoryPermissions() {
        String clusterDirPermsOct = this.configuration.get("slider.cluster.directory.permissions", "750");
        return new FsPermission(clusterDirPermsOct);
    }

    public void verifyClusterDirectoryNonexistent(String clustername, Path clusterDirectory) throws IOException, SliderException {
        if (this.fileSystem.exists(clusterDirectory)) {
            throw new SliderException(75, "Application Instance \"%s\" already exists and is defined in %s", clustername, clusterDirectory);
        }
    }

    public void verifyDirectoryNonexistent(Path clusterDirectory) throws IOException, SliderException {
        if (this.fileSystem.exists(clusterDirectory)) {
            log.error("Dir {} exists: {}", (Object)clusterDirectory, (Object)this.listFSDir(clusterDirectory));
            throw new SliderException(75, "Application Instance dir already exists: %s", clusterDirectory);
        }
    }

    public void verifyDirectoryWriteAccess(Path dirPath) throws IOException, SliderException {
        this.verifyPathExists(dirPath);
        Path tempFile = new Path(dirPath, "tmp-file-for-checks");
        try {
            FSDataOutputStream out = this.fileSystem.create(tempFile, true);
            IOUtils.closeStream((Closeable)out);
            this.fileSystem.delete(tempFile, false);
        }
        catch (IOException e) {
            log.warn("Failed to create file {}: {}", (Object)tempFile, (Object)e);
            throw new BadClusterStateException(e, "Unable to write to directory %s : %s", dirPath, e.toString());
        }
    }

    public void verifyPathExists(Path path) throws IOException {
        if (!this.fileSystem.exists(path)) {
            throw new FileNotFoundException(path.toString());
        }
    }

    public void verifyFileExists(Path path) throws IOException {
        FileStatus status = this.fileSystem.getFileStatus(path);
        if (!status.isFile()) {
            throw new FileNotFoundException("Not a file: " + path.toString());
        }
    }

    public boolean isFile(Path path) {
        boolean isFile = false;
        try {
            FileStatus status = this.fileSystem.getFileStatus(path);
            if (status.isFile()) {
                isFile = true;
            }
        }
        catch (IOException iOException) {}
        return isFile;
    }

    public void verifyFileExistsInZip(Path path, String file) throws IOException {
        this.fileSystem.copyToLocalFile(path, new Path("/tmp"));
        File dst = new File(new Path("/tmp", path.getName()).toString());
        boolean found = false;
        try (ZipFile zipFile = new ZipFile(dst);){
            Enumeration<? extends ZipEntry> entries = zipFile.entries();
            while (entries.hasMoreElements()) {
                ZipEntry entry = entries.nextElement();
                String nm = entry.getName();
                if (!nm.endsWith(file)) continue;
                found = true;
                break;
            }
        }
        dst.delete();
        if (!found) {
            throw new FileNotFoundException("file: " + file + " not found in " + path);
        }
        log.info("Verification of " + path + " passed");
    }

    public Path createAppInstanceTempPath(String clustername, String subdir) throws IOException {
        Path tmp = this.getTempPathForCluster(clustername);
        Path instancePath = new Path(tmp, subdir);
        this.fileSystem.mkdirs(instancePath);
        return instancePath;
    }

    public Path purgeAppInstanceTempFiles(String clustername) throws IOException {
        Path tmp = this.getTempPathForCluster(clustername);
        this.fileSystem.delete(tmp, true);
        return tmp;
    }

    public Path getBaseApplicationPath() {
        String configuredBasePath = this.configuration.get("slider.base.path");
        return configuredBasePath != null ? new Path(configuredBasePath) : new Path(this.getHomeDirectory(), ".slider");
    }

    public Path getDependencyPath() {
        String parentDir = SliderUtils.isHdp() ? "/hdp/apps/%s/slider" : "/apps/%s/slider";
        Path dependencyPath = new Path(String.format(parentDir, SliderUtils.getSliderVersion()));
        return dependencyPath;
    }

    public Path getDependencyTarGzip() {
        Path dependencyLibAmPath = this.getDependencyPath();
        Path dependencyLibTarGzip = new Path(dependencyLibAmPath.toUri().toString(), "slider.tar.gz");
        return dependencyLibTarGzip;
    }

    public Path getHomeDirectory() {
        return this.fileSystem.getHomeDirectory();
    }

    public boolean maybeAddImagePath(Map<String, LocalResource> localResources, Path imagePath) throws IOException {
        if (imagePath != null) {
            LocalResource resource = this.createAmResource(imagePath, LocalResourceType.ARCHIVE);
            localResources.put("expandedarchive", resource);
            return true;
        }
        return false;
    }

    public boolean maybeAddImagePath(Map<String, LocalResource> localResources, String imagePath) throws IOException {
        return imagePath != null && this.maybeAddImagePath(localResources, new Path(imagePath));
    }

    public LocalResource createAmResource(Path destPath, LocalResourceType resourceType) throws IOException {
        FileStatus destStatus = this.fileSystem.getFileStatus(destPath);
        LocalResource amResource = (LocalResource)Records.newRecord(LocalResource.class);
        amResource.setType(resourceType);
        amResource.setVisibility(LocalResourceVisibility.APPLICATION);
        amResource.setResource(ConverterUtils.getYarnUrlFromPath((Path)this.fileSystem.resolvePath(destStatus.getPath())));
        amResource.setTimestamp(destStatus.getModificationTime());
        amResource.setSize(destStatus.getLen());
        return amResource;
    }

    public Map<String, LocalResource> submitDirectory(Path srcDir, String destRelativeDir) throws IOException {
        FileStatus[] fileset = this.fileSystem.listStatus(srcDir);
        HashMap<String, LocalResource> localResources = new HashMap<String, LocalResource>(fileset.length);
        FileStatus[] fileStatusArray = fileset;
        int n = fileset.length;
        int n2 = 0;
        while (n2 < n) {
            FileStatus entry = fileStatusArray[n2];
            LocalResource resource = this.createAmResource(entry.getPath(), LocalResourceType.FILE);
            String relativePath = String.valueOf(destRelativeDir) + "/" + entry.getPath().getName();
            localResources.put(relativePath, resource);
            ++n2;
        }
        return localResources;
    }

    public LocalResource submitJarWithClass(Class clazz, Path tempPath, String subdir, String jarName) throws IOException, SliderException {
        File localFile = SliderUtils.findContainingJarOrFail(clazz);
        return this.submitFile(localFile, tempPath, subdir, jarName);
    }

    public LocalResource submitFile(File localFile, Path tempPath, String subdir, String destFileName) throws IOException {
        Path src = new Path(localFile.toString());
        Path subdirPath = new Path(tempPath, subdir);
        this.fileSystem.mkdirs(subdirPath);
        Path destPath = new Path(subdirPath, destFileName);
        log.debug("Copying {} (size={} bytes) to {}", new Object[]{localFile, localFile.length(), destPath});
        this.fileSystem.copyFromLocalFile(false, true, src, destPath);
        return this.createAmResource(destPath, LocalResourceType.FILE);
    }

    public void submitTarGzipAndUpdate(Map<String, LocalResource> providerResources) throws IOException, BadClusterStateException {
        Path dependencyLibTarGzip = this.getDependencyTarGzip();
        LocalResource lc = this.createAmResource(dependencyLibTarGzip, LocalResourceType.ARCHIVE);
        providerResources.put("slider_dep", lc);
    }

    public void copyLocalFilesToHdfs(File localPath, FilenameFilter filenameFilter, Path destDir, FsPermission fp) throws IOException {
        if (localPath == null || destDir == null) {
            throw new IOException("Either localPath or destDir is null");
        }
        this.fileSystem.getConf().set("fs.permissions.umask-mode", "000");
        this.fileSystem.mkdirs(destDir, fp);
        if (localPath.isDirectory()) {
            File[] localFiles = localPath.listFiles(filenameFilter);
            Path[] localFilePaths = new Path[localFiles.length];
            int i = 0;
            File[] fileArray = localFiles;
            int n = localFiles.length;
            int n2 = 0;
            while (n2 < n) {
                File localFile = fileArray[n2];
                localFilePaths[i++] = new Path(localFile.getPath());
                ++n2;
            }
            log.info("Copying {} files from {} to {}", new Object[]{i, localPath.toURI(), destDir.toUri()});
            this.fileSystem.copyFromLocalFile(false, true, localFilePaths, destDir);
        } else {
            log.info("Copying file {} to {}", (Object)localPath.toURI(), (Object)destDir.toUri());
            this.fileSystem.copyFromLocalFile(false, true, new Path(localPath.getPath()), destDir);
        }
        this.fileSystem.setPermission(destDir, fp);
    }

    public void copyLocalFileToHdfs(File localPath, Path destPath, FsPermission fp) throws IOException {
        if (localPath == null || destPath == null) {
            throw new IOException("Either localPath or destPath is null");
        }
        this.fileSystem.getConf().set("fs.permissions.umask-mode", "000");
        this.fileSystem.mkdirs(destPath.getParent(), fp);
        log.info("Copying file {} to {}", (Object)localPath.toURI(), (Object)destPath.toUri());
        this.fileSystem.copyFromLocalFile(false, true, new Path(localPath.getPath()), destPath);
        this.fileSystem.setPermission(destPath, fp);
    }

    public String listFSDir(Path path) throws IOException {
        FileStatus[] stats = this.fileSystem.listStatus(path);
        StringBuilder builder = new StringBuilder();
        FileStatus[] fileStatusArray = stats;
        int n = stats.length;
        int n2 = 0;
        while (n2 < n) {
            FileStatus stat = fileStatusArray[n2];
            builder.append(stat.getPath().toString()).append("\t").append(stat.getLen()).append("\n");
            ++n2;
        }
        return builder.toString();
    }

    public Map<String, Path> listPersistentInstances() throws IOException {
        FileSystem fs = this.getFileSystem();
        Path path = new Path(this.getBaseApplicationPath(), "cluster");
        log.debug("Looking for all persisted application at {}", (Object)path.toString());
        if (!fs.exists(path)) {
            return new HashMap<String, Path>(0);
        }
        FileStatus[] statuses = fs.listStatus(path);
        HashMap<String, Path> instances = new HashMap<String, Path>(statuses.length);
        FileStatus[] fileStatusArray = statuses;
        int n = statuses.length;
        int n2 = 0;
        while (n2 < n) {
            FileStatus status = fileStatusArray[n2];
            if (status.isDirectory()) {
                Path child = status.getPath();
                Path internalJson = new Path(child, "internal.json");
                if (fs.exists(internalJson)) {
                    instances.put(child.getName(), child);
                } else {
                    log.info("Malformed cluster found at {}. It does not appear to be a valid persisted instance.", (Object)child.toString());
                }
            }
            ++n2;
        }
        return instances;
    }

    public void touch(Path path, boolean overwrite) throws IOException {
        FSDataOutputStream out = this.fileSystem.create(path, overwrite);
        out.close();
    }

    public void cat(Path path, boolean overwrite, String data) throws IOException {
        FSDataOutputStream out = this.fileSystem.create(path, overwrite);
        byte[] bytes = data.getBytes(Charset.forName("UTF-8"));
        out.write(bytes);
        out.close();
    }

    public Path createPathThatMustExist(String uri) throws SliderException, IOException {
        Preconditions.checkNotNull((Object)uri);
        Path path = new Path(uri);
        this.verifyPathExists(path);
        return path;
    }

    public Path locateInstanceDefinition(String clustername) throws IOException, SliderException {
        Path clusterDirectory = this.buildClusterDirPath(clustername);
        Path appConfPath = new Path(clusterDirectory, "app_config.json");
        this.verifyClusterSpecExists(clustername, appConfPath);
        return appConfPath;
    }

    public void verifyClusterSpecExists(String clustername, Path clusterSpecPath) throws IOException, SliderException {
        if (!this.fileSystem.isFile(clusterSpecPath)) {
            log.debug("Missing specification file {}", (Object)clusterSpecPath);
            throw UnknownApplicationInstanceException.unknownInstance(String.valueOf(clustername) + "\n (definition not found at " + clusterSpecPath);
        }
    }
}

