/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hdfs.server.datanode;

import com.cloudera.org.codehaus.jackson.map.ObjectMapper;
import com.google.common.base.Preconditions;
import java.io.IOException;
import java.nio.charset.Charset;
import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.Future;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.locks.ReentrantLock;
import org.apache.commons.codec.digest.DigestUtils;
import org.apache.hadoop.classification.InterfaceAudience;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.hdfs.protocol.ExtendedBlock;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkItem;
import org.apache.hadoop.hdfs.server.datanode.DiskBalancerWorkStatus;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsDatasetSpi;
import org.apache.hadoop.hdfs.server.datanode.fsdataset.FsVolumeSpi;
import org.apache.hadoop.hdfs.server.diskbalancer.DiskBalancerException;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.NodePlan;
import org.apache.hadoop.hdfs.server.diskbalancer.planner.Step;
import org.apache.hadoop.util.Time;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class DiskBalancer {
    private static final Logger LOG = LoggerFactory.getLogger(DiskBalancer.class);
    private final FsDatasetSpi<?> dataset;
    private final String dataNodeUUID;
    private final BlockMover blockMover;
    private final ReentrantLock lock;
    private final ConcurrentHashMap<VolumePair, DiskBalancerWorkItem> workMap;
    private boolean isDiskBalancerEnabled = false;
    private ExecutorService scheduler;
    private Future future;
    private String planID;
    private DiskBalancerWorkStatus.Result currentResult = DiskBalancerWorkStatus.Result.NO_PLAN;
    private long bandwidth;

    public DiskBalancer(String dataNodeUUID, Configuration conf, BlockMover blockMover) {
        this.blockMover = blockMover;
        this.dataset = this.blockMover.getDataset();
        this.dataNodeUUID = dataNodeUUID;
        this.scheduler = Executors.newSingleThreadExecutor();
        this.lock = new ReentrantLock();
        this.workMap = new ConcurrentHashMap();
        this.isDiskBalancerEnabled = conf.getBoolean("dfs.disk.balancer.enabled", false);
        this.bandwidth = conf.getInt("dfs.disk.balancer.max.disk.throughputInMBperSec", 10);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void shutdown() {
        this.lock.lock();
        try {
            this.isDiskBalancerEnabled = false;
            this.currentResult = DiskBalancerWorkStatus.Result.NO_PLAN;
            if (this.future != null && !this.future.isDone()) {
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_CANCELLED;
                this.blockMover.setExitFlag();
                this.shutdownExecutor();
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    private void shutdownExecutor() {
        int secondsTowait = 10;
        this.scheduler.shutdown();
        try {
            if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                this.scheduler.shutdownNow();
                if (!this.scheduler.awaitTermination(10L, TimeUnit.SECONDS)) {
                    LOG.error("Disk Balancer : Scheduler did not terminate.");
                }
            }
        }
        catch (InterruptedException ex) {
            this.scheduler.shutdownNow();
            Thread.currentThread().interrupt();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void submitPlan(String planID, long planVersion, String plan, boolean force) throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            if (this.future != null && !this.future.isDone()) {
                LOG.error("Disk Balancer - Executing another plan, submitPlan failed.");
                throw new DiskBalancerException("Executing another plan", DiskBalancerException.Result.PLAN_ALREADY_IN_PROGRESS);
            }
            NodePlan nodePlan = this.verifyPlan(planID, planVersion, plan, force);
            this.createWorkPlan(nodePlan);
            this.planID = planID;
            this.currentResult = DiskBalancerWorkStatus.Result.PLAN_UNDER_PROGRESS;
            this.executePlan();
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public DiskBalancerWorkStatus queryWorkStatus() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            if (this.currentResult == DiskBalancerWorkStatus.Result.PLAN_UNDER_PROGRESS && this.future != null && this.future.isDone()) {
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_DONE;
            }
            DiskBalancerWorkStatus status = new DiskBalancerWorkStatus(this.currentResult, this.planID);
            for (Map.Entry<VolumePair, DiskBalancerWorkItem> entry : this.workMap.entrySet()) {
                DiskBalancerWorkStatus.DiskBalancerWorkEntry workEntry = new DiskBalancerWorkStatus.DiskBalancerWorkEntry(entry.getKey().getSource().getBasePath(), entry.getKey().getDest().getBasePath(), entry.getValue());
                status.addWorkEntry(workEntry);
            }
            DiskBalancerWorkStatus diskBalancerWorkStatus = status;
            return diskBalancerWorkStatus;
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cancelPlan(String planID) throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            if (this.planID == null || !this.planID.equals(planID)) {
                LOG.error("Disk Balancer - No such plan. Cancel plan failed. PlanID: " + planID);
                throw new DiskBalancerException("No such plan.", DiskBalancerException.Result.NO_SUCH_PLAN);
            }
            if (!this.future.isDone()) {
                this.blockMover.setExitFlag();
                this.shutdownExecutor();
                this.currentResult = DiskBalancerWorkStatus.Result.PLAN_CANCELLED;
            }
        }
        finally {
            this.lock.unlock();
        }
    }

    public String getVolumeNames() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            HashMap<String, String> pathMap = new HashMap<String, String>();
            Map<String, FsVolumeSpi> volMap = this.getStorageIDToVolumeMap();
            for (Map.Entry<String, FsVolumeSpi> entry : volMap.entrySet()) {
                pathMap.put(entry.getKey(), entry.getValue().getBasePath());
            }
            ObjectMapper mapper = new ObjectMapper();
            String string = mapper.writeValueAsString(pathMap);
            return string;
        }
        catch (IOException e) {
            throw new DiskBalancerException("Internal error, Unable to create JSON string.", e, DiskBalancerException.Result.INTERNAL_ERROR);
        }
        finally {
            this.lock.unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public long getBandwidth() throws DiskBalancerException {
        this.lock.lock();
        try {
            this.checkDiskBalancerEnabled();
            long l = this.bandwidth;
            return l;
        }
        finally {
            this.lock.unlock();
        }
    }

    private void checkDiskBalancerEnabled() throws DiskBalancerException {
        if (!this.isDiskBalancerEnabled) {
            LOG.error("Disk Balancer is not enabled.");
            throw new DiskBalancerException("Disk Balancer is not enabled.", DiskBalancerException.Result.DISK_BALANCER_NOT_ENABLED);
        }
    }

    private NodePlan verifyPlan(String planID, long planVersion, String plan, boolean force) throws DiskBalancerException {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.verifyPlanVersion(planVersion);
        NodePlan nodePlan = this.verifyPlanHash(planID, plan);
        if (!force) {
            this.verifyTimeStamp(nodePlan);
        }
        this.verifyNodeUUID(nodePlan);
        return nodePlan;
    }

    private void verifyPlanVersion(long planVersion) throws DiskBalancerException {
        if (planVersion < 1L || planVersion > 1L) {
            LOG.error("Disk Balancer - Invalid plan version.");
            throw new DiskBalancerException("Invalid plan version.", DiskBalancerException.Result.INVALID_PLAN_VERSION);
        }
    }

    private NodePlan verifyPlanHash(String planID, String plan) throws DiskBalancerException {
        long sha512Length = 128L;
        if (plan == null || plan.length() == 0) {
            LOG.error("Disk Balancer -  Invalid plan.");
            throw new DiskBalancerException("Invalid plan.", DiskBalancerException.Result.INVALID_PLAN);
        }
        if (planID == null || (long)planID.length() != 128L || !DigestUtils.sha512Hex((byte[])plan.getBytes(Charset.forName("UTF-8"))).equalsIgnoreCase(planID)) {
            LOG.error("Disk Balancer - Invalid plan hash.");
            throw new DiskBalancerException("Invalid or mis-matched hash.", DiskBalancerException.Result.INVALID_PLAN_HASH);
        }
        try {
            return NodePlan.parseJson(plan);
        }
        catch (IOException ex) {
            throw new DiskBalancerException("Parsing plan failed.", ex, DiskBalancerException.Result.MALFORMED_PLAN);
        }
    }

    private void verifyTimeStamp(NodePlan plan) throws DiskBalancerException {
        long now = Time.now();
        long planTime = plan.getTimeStamp();
        if (planTime + TimeUnit.HOURS.toMillis(24L) < now) {
            String hourString = "Plan was generated more than " + Integer.toString(24) + " hours ago.";
            LOG.error("Disk Balancer - " + hourString);
            throw new DiskBalancerException(hourString, DiskBalancerException.Result.OLD_PLAN_SUBMITTED);
        }
    }

    private void verifyNodeUUID(NodePlan plan) throws DiskBalancerException {
        if (plan.getNodeUUID() == null || !plan.getNodeUUID().equals(this.dataNodeUUID)) {
            LOG.error("Disk Balancer - Plan was generated for another node.");
            throw new DiskBalancerException("Plan was generated for another node.", DiskBalancerException.Result.DATANODE_ID_MISMATCH);
        }
    }

    private void createWorkPlan(NodePlan plan) throws DiskBalancerException {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.workMap.clear();
        Map<String, FsVolumeSpi> pathMap = this.getStorageIDToVolumeMap();
        for (Step step : plan.getVolumeSetPlans()) {
            String sourceuuid = step.getSourceVolume().getUuid();
            String destinationuuid = step.getDestinationVolume().getUuid();
            FsVolumeSpi sourceVol = pathMap.get(sourceuuid);
            if (sourceVol == null) {
                LOG.error("Disk Balancer - Unable to find source volume. submitPlan failed.");
                throw new DiskBalancerException("Unable to find source volume.", DiskBalancerException.Result.INVALID_VOLUME);
            }
            FsVolumeSpi destVol = pathMap.get(destinationuuid);
            if (destVol == null) {
                LOG.error("Disk Balancer - Unable to find destination volume. submitPlan failed.");
                throw new DiskBalancerException("Unable to find destination volume.", DiskBalancerException.Result.INVALID_VOLUME);
            }
            this.createWorkPlan(sourceVol, destVol, step);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private Map<String, FsVolumeSpi> getStorageIDToVolumeMap() throws DiskBalancerException {
        HashMap<String, FsVolumeSpi> pathMap = new HashMap<String, FsVolumeSpi>();
        try {
            FsDatasetSpi<?> fsDatasetSpi = this.dataset;
            synchronized (fsDatasetSpi) {
                FsDatasetSpi.FsVolumeReferences references = this.dataset.getFsVolumeReferences();
                for (int ndx = 0; ndx < references.size(); ++ndx) {
                    FsVolumeSpi vol = references.get(ndx);
                    pathMap.put(vol.getStorageID(), vol);
                }
                references.close();
            }
        }
        catch (IOException ex) {
            LOG.error("Disk Balancer - Internal Error.", (Throwable)ex);
            throw new DiskBalancerException("Internal error", ex, DiskBalancerException.Result.INTERNAL_ERROR);
        }
        return pathMap;
    }

    private void executePlan() {
        Preconditions.checkState((boolean)this.lock.isHeldByCurrentThread());
        this.blockMover.setRunnable();
        if (this.scheduler.isShutdown()) {
            this.scheduler = Executors.newSingleThreadExecutor();
        }
        this.future = this.scheduler.submit(new Runnable(){

            @Override
            public void run() {
                Thread.currentThread().setName("DiskBalancerThread");
                LOG.info("Executing Disk balancer plan. Plan ID -  " + DiskBalancer.this.planID);
                for (Map.Entry entry : DiskBalancer.this.workMap.entrySet()) {
                    DiskBalancer.this.blockMover.copyBlocks((VolumePair)entry.getKey(), (DiskBalancerWorkItem)entry.getValue());
                }
            }
        });
    }

    private void createWorkPlan(FsVolumeSpi source, FsVolumeSpi dest, Step step) throws DiskBalancerException {
        if (source.getStorageID().equals(dest.getStorageID())) {
            LOG.info("Disk Balancer - source & destination volumes are same.");
            throw new DiskBalancerException("source and destination volumes are same.", DiskBalancerException.Result.INVALID_MOVE);
        }
        VolumePair pair = new VolumePair(source, dest);
        long bytesToMove = step.getBytesToMove();
        if (this.workMap.containsKey(pair)) {
            bytesToMove += this.workMap.get(pair).getBytesToCopy();
        }
        DiskBalancerWorkItem work = new DiskBalancerWorkItem(bytesToMove, 0L);
        work.setBandwidth(step.getBandwidth());
        work.setTolerancePercent(step.getTolerancePercent());
        work.setMaxDiskErrors(step.getMaxDiskErrors());
        this.workMap.put(pair, work);
    }

    public static class DiskBalancerMover
    implements BlockMover {
        private final FsDatasetSpi dataset;
        private long diskBandwidth;
        private long blockTolerance;
        private long maxDiskErrors;
        private int poolIndex;
        private AtomicBoolean shouldRun;

        public DiskBalancerMover(FsDatasetSpi dataset, Configuration conf) {
            this.dataset = dataset;
            this.shouldRun = new AtomicBoolean(false);
            this.diskBandwidth = conf.getLong("dfs.disk.balancer.max.disk.throughputInMBperSec", 10L);
            this.blockTolerance = conf.getLong("dfs.disk.balancer.block.tolerance.percent", 5L);
            this.maxDiskErrors = conf.getLong("dfs.disk.balancer.max.disk.errors", 5L);
            if (this.diskBandwidth <= 0L) {
                LOG.debug("Found 0 or less as max disk throughput, ignoring config value. value : " + this.diskBandwidth);
                this.diskBandwidth = 10L;
            }
            if (this.blockTolerance <= 0L) {
                LOG.debug("Found 0 or less for block tolerance value, ignoring configvalue. value : " + this.blockTolerance);
                this.blockTolerance = 5L;
            }
            if (this.maxDiskErrors < 0L) {
                LOG.debug("Found  less than 0 for maxDiskErrors value, ignoring config value. value : " + this.maxDiskErrors);
                this.maxDiskErrors = 5L;
            }
        }

        @Override
        public void setRunnable() {
            this.shouldRun.set(true);
        }

        @Override
        public void setExitFlag() {
            this.shouldRun.set(false);
        }

        public boolean shouldRun() {
            return this.shouldRun.get();
        }

        private boolean isLessThanNeeded(long blockSize, DiskBalancerWorkItem item) {
            long bytesToCopy = item.getBytesToCopy() - item.getBytesCopied();
            return blockSize <= (bytesToCopy += bytesToCopy * this.getBlockTolerancePercentage(item) / 100L);
        }

        private long getBlockTolerancePercentage(DiskBalancerWorkItem item) {
            return item.getTolerancePercent() <= 0L ? this.blockTolerance : item.getTolerancePercent();
        }

        private boolean isCloseEnough(DiskBalancerWorkItem item) {
            long temp = item.getBytesCopied() + item.getBytesCopied() * this.getBlockTolerancePercentage(item) / 100L;
            return item.getBytesToCopy() < temp;
        }

        private long getDiskBandwidth(DiskBalancerWorkItem item) {
            return item.getBandwidth() <= 0L ? this.diskBandwidth : item.getBandwidth();
        }

        private long computeDelay(long bytesCopied, long timeUsed, DiskBalancerWorkItem item) {
            if (timeUsed == 0L) {
                return 0L;
            }
            int megaByte = 0x100000;
            long bytesInMB = bytesCopied / 0x100000L;
            long lastThroughput = bytesInMB / TimeUnit.SECONDS.convert(timeUsed, TimeUnit.MILLISECONDS);
            long delay = bytesInMB / this.getDiskBandwidth(item) - lastThroughput;
            return delay <= 0L ? 0L : TimeUnit.MILLISECONDS.convert(delay, TimeUnit.SECONDS);
        }

        private long getMaxError(DiskBalancerWorkItem item) {
            return item.getMaxDiskErrors() <= 0L ? this.maxDiskErrors : item.getMaxDiskErrors();
        }

        private ExtendedBlock getBlockToCopy(FsVolumeSpi.BlockIterator iter, DiskBalancerWorkItem item) {
            while (!iter.atEnd() && item.getErrorCount() < this.getMaxError(item)) {
                try {
                    ExtendedBlock block = iter.nextBlock();
                    if (!this.dataset.isValidBlock(block) || !this.isLessThanNeeded(block.getNumBytes(), item)) continue;
                    return block;
                }
                catch (IOException e) {
                    item.incErrorCount();
                }
            }
            if (item.getErrorCount() >= this.getMaxError(item)) {
                item.setErrMsg("Error count exceeded.");
                LOG.info("Maximum error count exceeded. Error count: {} Max error:{} ", (Object)item.getErrorCount(), (Object)item.getMaxDiskErrors());
            }
            return null;
        }

        private void openPoolIters(FsVolumeSpi source, List<FsVolumeSpi.BlockIterator> poolIters) {
            Preconditions.checkNotNull((Object)source);
            Preconditions.checkNotNull(poolIters);
            for (String blockPoolID : source.getBlockPoolList()) {
                poolIters.add(source.newBlockIterator(blockPoolID, "DiskBalancerSource"));
            }
        }

        ExtendedBlock getNextBlock(List<FsVolumeSpi.BlockIterator> poolIters, DiskBalancerWorkItem item) {
            Preconditions.checkNotNull(poolIters);
            ExtendedBlock block = null;
            for (int currentCount = 0; block == null && currentCount < poolIters.size(); ++currentCount) {
                this.poolIndex = this.poolIndex++ % poolIters.size();
                FsVolumeSpi.BlockIterator currentPoolIter = poolIters.get(this.poolIndex);
                block = this.getBlockToCopy(currentPoolIter, item);
            }
            if (block == null) {
                try {
                    item.setErrMsg("No source blocks found to move.");
                    LOG.error("No movable source blocks found. {}", (Object)item.toJson());
                }
                catch (IOException e) {
                    LOG.error("Unable to get json from Item.");
                }
            }
            return block;
        }

        private void closePoolIters(List<FsVolumeSpi.BlockIterator> poolIters) {
            Preconditions.checkNotNull(poolIters);
            for (FsVolumeSpi.BlockIterator iter : poolIters) {
                try {
                    iter.close();
                }
                catch (IOException ex) {
                    LOG.error("Error closing a block pool iter. ex: {}", (Throwable)ex);
                }
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        @Override
        public void copyBlocks(VolumePair pair, DiskBalancerWorkItem item) {
            FsVolumeSpi source = pair.getSource();
            FsVolumeSpi dest = pair.getDest();
            LinkedList<FsVolumeSpi.BlockIterator> poolIters = new LinkedList<FsVolumeSpi.BlockIterator>();
            if (source.isTransientStorage() || dest.isTransientStorage()) {
                return;
            }
            try {
                this.openPoolIters(source, poolIters);
                if (poolIters.size() == 0) {
                    LOG.error("No block pools found on volume. volume : {}. Exiting.", (Object)source.getBasePath());
                    return;
                }
                while (this.shouldRun()) {
                    try {
                        if (item.getErrorCount() > this.getMaxError(item)) {
                            LOG.error("Exceeded the max error count. source {}, dest: {} error count: {}", new Object[]{source.getBasePath(), dest.getBasePath(), item.getErrorCount()});
                            this.setExitFlag();
                            continue;
                        }
                        if (this.isCloseEnough(item)) {
                            LOG.info("Copy from {} to {} done. copied {} bytes and {} blocks.", new Object[]{source.getBasePath(), dest.getBasePath(), item.getBytesCopied(), item.getBlocksCopied()});
                            this.setExitFlag();
                            continue;
                        }
                        ExtendedBlock block = this.getNextBlock(poolIters, item);
                        if (block == null) {
                            this.setExitFlag();
                            LOG.error("No source blocks, exiting the copy. Source: {}, dest:{}", (Object)source.getBasePath(), (Object)dest.getBasePath());
                            continue;
                        }
                        if (!this.shouldRun()) continue;
                        if (dest.getAvailable() <= item.getBytesToCopy()) {
                            LOG.error("Destination volume: {} does not have enough space to accommodate a block. Block Size: {} Exiting from copyBlocks.", (Object)dest.getBasePath(), (Object)block.getNumBytes());
                            this.setExitFlag();
                            continue;
                        }
                        long begin = System.nanoTime();
                        this.dataset.moveBlockAcrossVolumes(block, dest);
                        long now = System.nanoTime();
                        long timeUsed = now - begin > 0L ? now - begin : 0L;
                        LOG.debug("Moved block with size {} from  {} to {}", new Object[]{block.getNumBytes(), source.getBasePath(), dest.getBasePath()});
                        item.incCopiedSoFar(block.getNumBytes());
                        item.incBlocksCopied();
                        Thread.sleep(this.computeDelay(block.getNumBytes(), timeUsed, item));
                    }
                    catch (IOException ex) {
                        LOG.error("Exception while trying to copy blocks. error: {}", (Throwable)ex);
                        item.incErrorCount();
                    }
                    catch (InterruptedException e) {
                        LOG.error("Copy Block Thread interrupted, exiting the copy.");
                        Thread.currentThread().interrupt();
                        item.incErrorCount();
                        this.setExitFlag();
                    }
                }
            }
            finally {
                this.closePoolIters(poolIters);
            }
        }

        @Override
        public FsDatasetSpi getDataset() {
            return this.dataset;
        }
    }

    public static class VolumePair {
        private final FsVolumeSpi source;
        private final FsVolumeSpi dest;

        public VolumePair(FsVolumeSpi source, FsVolumeSpi dest) {
            this.source = source;
            this.dest = dest;
        }

        public FsVolumeSpi getSource() {
            return this.source;
        }

        public FsVolumeSpi getDest() {
            return this.dest;
        }

        public boolean equals(Object o) {
            if (this == o) {
                return true;
            }
            if (o == null || this.getClass() != o.getClass()) {
                return false;
            }
            VolumePair that = (VolumePair)o;
            return this.source.equals(that.source) && this.dest.equals(that.dest);
        }

        public int hashCode() {
            int result = this.source.getBasePath().hashCode();
            result = 31 * result + this.dest.getBasePath().hashCode();
            return result;
        }
    }

    public static interface BlockMover {
        public void copyBlocks(VolumePair var1, DiskBalancerWorkItem var2);

        public void setRunnable();

        public void setExitFlag();

        public FsDatasetSpi getDataset();
    }
}

