/*
 * Decompiled with CFR 0.152.
 */
package org.apache.hadoop.hbase.master.cleaner;

import java.io.IOException;
import java.util.ArrayList;
import java.util.List;
import java.util.concurrent.CountDownLatch;
import java.util.concurrent.LinkedBlockingQueue;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import org.apache.hadoop.conf.Configuration;
import org.apache.hadoop.fs.FileStatus;
import org.apache.hadoop.fs.FileSystem;
import org.apache.hadoop.fs.Path;
import org.apache.hadoop.hbase.Stoppable;
import org.apache.hadoop.hbase.master.cleaner.BaseLogCleanerDelegate;
import org.apache.hadoop.hbase.master.cleaner.CleanerChore;
import org.apache.hadoop.hbase.master.procedure.MasterProcedureUtil;
import org.apache.hadoop.hbase.wal.AbstractFSWALProvider;
import org.apache.hbase.thirdparty.com.google.common.annotations.VisibleForTesting;
import org.apache.hbase.thirdparty.com.google.common.base.Preconditions;
import org.apache.yetus.audience.InterfaceAudience;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

@InterfaceAudience.Private
public class LogCleaner
extends CleanerChore<BaseLogCleanerDelegate> {
    private static final Logger LOG = LoggerFactory.getLogger(LogCleaner.class);
    public static final String OLD_WALS_CLEANER_THREAD_SIZE = "hbase.oldwals.cleaner.thread.size";
    public static final int DEFAULT_OLD_WALS_CLEANER_THREAD_SIZE = 2;
    public static final String OLD_WALS_CLEANER_THREAD_TIMEOUT_MSEC = "hbase.oldwals.cleaner.thread.timeout.msec";
    @VisibleForTesting
    static final long DEFAULT_OLD_WALS_CLEANER_THREAD_TIMEOUT_MSEC = 60000L;
    private final LinkedBlockingQueue<CleanerContext> pendingDelete = new LinkedBlockingQueue();
    private List<Thread> oldWALsCleaner;
    private long cleanerThreadTimeoutMsec;

    public LogCleaner(int period, Stoppable stopper, Configuration conf, FileSystem fs, Path oldLogDir) {
        super("LogsCleaner", period, stopper, conf, fs, oldLogDir, "hbase.master.logcleaner.plugins");
        int size = conf.getInt(OLD_WALS_CLEANER_THREAD_SIZE, 2);
        this.oldWALsCleaner = this.createOldWalsCleaner(size);
        this.cleanerThreadTimeoutMsec = conf.getLong(OLD_WALS_CLEANER_THREAD_TIMEOUT_MSEC, 60000L);
    }

    @Override
    protected boolean validate(Path file) {
        return AbstractFSWALProvider.validateWALFilename(file.getName()) || MasterProcedureUtil.validateProcedureWALFilename(file.getName());
    }

    @Override
    public void onConfigurationChange(Configuration conf) {
        super.onConfigurationChange(conf);
        int newSize = conf.getInt(OLD_WALS_CLEANER_THREAD_SIZE, 2);
        if (newSize == this.oldWALsCleaner.size()) {
            LOG.debug("Size from configuration is the same as previous which is {}, no need to update.", (Object)newSize);
            return;
        }
        this.interruptOldWALsCleaner();
        this.oldWALsCleaner = this.createOldWalsCleaner(newSize);
        this.cleanerThreadTimeoutMsec = conf.getLong(OLD_WALS_CLEANER_THREAD_TIMEOUT_MSEC, 60000L);
    }

    @Override
    protected int deleteFiles(Iterable<FileStatus> filesToDelete) {
        ArrayList<CleanerContext> results = new ArrayList<CleanerContext>();
        for (FileStatus file : filesToDelete) {
            LOG.trace("Scheduling file {} for deletion", (Object)file);
            if (file == null) continue;
            results.add(new CleanerContext(file));
        }
        LOG.debug("Old WAL files pending deletion: {}", results);
        this.pendingDelete.addAll(results);
        int deletedFiles = 0;
        for (CleanerContext res : results) {
            LOG.trace("Awaiting the results for deletion of old WAL file: {}", (Object)res);
            deletedFiles += res.getResult(this.cleanerThreadTimeoutMsec) ? 1 : 0;
        }
        return deletedFiles;
    }

    @Override
    public synchronized void cleanup() {
        super.cleanup();
        this.interruptOldWALsCleaner();
    }

    @VisibleForTesting
    int getSizeOfCleaners() {
        return this.oldWALsCleaner.size();
    }

    @VisibleForTesting
    long getCleanerThreadTimeoutMsec() {
        return this.cleanerThreadTimeoutMsec;
    }

    private List<Thread> createOldWalsCleaner(int size) {
        LOG.info("Creating {} OldWALs cleaner threads", (Object)size);
        ArrayList<Thread> oldWALsCleaner = new ArrayList<Thread>(size);
        for (int i = 0; i < size; ++i) {
            Thread cleaner = new Thread(() -> this.deleteFile());
            cleaner.setName("OldWALsCleaner-" + i);
            cleaner.setDaemon(true);
            cleaner.start();
            oldWALsCleaner.add(cleaner);
        }
        return oldWALsCleaner;
    }

    private void interruptOldWALsCleaner() {
        for (Thread cleaner : this.oldWALsCleaner) {
            LOG.trace("Interrupting thread: {}", (Object)cleaner);
            cleaner.interrupt();
        }
        this.oldWALsCleaner.clear();
    }

    private void deleteFile() {
        while (true) {
            try {
                CleanerContext context = this.pendingDelete.take();
                Preconditions.checkNotNull((Object)context);
                FileStatus oldWalFile = context.getTargetToClean();
                try {
                    LOG.debug("Attempting to delete old WAL file: {}", (Object)oldWalFile);
                    boolean succeed = this.fs.delete(oldWalFile.getPath(), false);
                    context.setResult(succeed);
                }
                catch (IOException e) {
                    LOG.warn("Failed to clean old WAL file", (Throwable)e);
                    context.setResult(false);
                }
            }
            catch (InterruptedException ite) {
                LOG.warn("Interrupted while cleaning old WALs, will try to clean it next round. Exiting.");
                Thread.currentThread().interrupt();
                return;
            }
            LOG.debug("Exiting");
        }
    }

    public synchronized void cancel(boolean mayInterruptIfRunning) {
        LOG.debug("Cancelling LogCleaner");
        super.cancel(mayInterruptIfRunning);
        this.interruptOldWALsCleaner();
    }

    private static final class CleanerContext {
        final FileStatus target;
        final AtomicBoolean result;
        final CountDownLatch remainingResults;

        private CleanerContext(FileStatus status) {
            this.target = status;
            this.result = new AtomicBoolean(false);
            this.remainingResults = new CountDownLatch(1);
        }

        void setResult(boolean res) {
            this.result.set(res);
            this.remainingResults.countDown();
        }

        boolean getResult(long waitIfNotFinished) {
            try {
                boolean completed = this.remainingResults.await(waitIfNotFinished, TimeUnit.MILLISECONDS);
                if (!completed) {
                    LOG.warn("Spend too much time [{}ms] to delete old WAL file: {}", (Object)waitIfNotFinished, (Object)this.target);
                    return false;
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Interrupted while awaiting deletion of WAL file: {}", (Object)this.target);
                return false;
            }
            return this.result.get();
        }

        FileStatus getTargetToClean() {
            return this.target;
        }

        public String toString() {
            return "CleanerContext [target=" + this.target + ", result=" + this.result + "]";
        }
    }
}

