/*
 * Decompiled with CFR 0.152.
 */
package org.apache.cassandra.db.compaction;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Predicate;
import com.google.common.base.Predicates;
import com.google.common.collect.ImmutableSortedSet;
import com.google.common.collect.Iterables;
import com.google.common.collect.Sets;
import com.google.common.primitives.Ints;
import java.io.IOException;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.SortedSet;
import org.apache.cassandra.config.DatabaseDescriptor;
import org.apache.cassandra.db.ColumnFamilyStore;
import org.apache.cassandra.db.DecoratedKey;
import org.apache.cassandra.db.PartitionPosition;
import org.apache.cassandra.db.compaction.AbstractCompactionStrategy;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategy;
import org.apache.cassandra.db.compaction.SizeTieredCompactionStrategyOptions;
import org.apache.cassandra.dht.Bounds;
import org.apache.cassandra.dht.Range;
import org.apache.cassandra.dht.Token;
import org.apache.cassandra.io.sstable.Component;
import org.apache.cassandra.io.sstable.format.SSTableReader;
import org.apache.cassandra.service.StorageService;
import org.apache.cassandra.utils.FBUtilities;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class LeveledManifest {
    private static final Logger logger = LoggerFactory.getLogger(LeveledManifest.class);
    private static final int MAX_COMPACTING_L0 = 32;
    private static final int NO_COMPACTION_LIMIT = 25;
    public static final int MAX_LEVEL_COUNT = (int)Math.log10(1.0E9);
    private final ColumnFamilyStore cfs;
    @VisibleForTesting
    protected final List<SSTableReader>[] generations;
    private final PartitionPosition[] lastCompactedKeys;
    private final long maxSSTableSizeInBytes;
    private final SizeTieredCompactionStrategyOptions options;
    private final int[] compactionCounter;
    private final int levelFanoutSize;
    private static final Predicate<SSTableReader> suspectP = new Predicate<SSTableReader>(){

        public boolean apply(SSTableReader candidate) {
            return candidate.isMarkedSuspect();
        }
    };

    LeveledManifest(ColumnFamilyStore cfs, int maxSSTableSizeInMB, int fanoutSize, SizeTieredCompactionStrategyOptions options) {
        this.cfs = cfs;
        this.maxSSTableSizeInBytes = (long)maxSSTableSizeInMB * 1024L * 1024L;
        this.options = options;
        this.levelFanoutSize = fanoutSize;
        this.generations = new List[MAX_LEVEL_COUNT];
        this.lastCompactedKeys = new PartitionPosition[MAX_LEVEL_COUNT];
        for (int i = 0; i < this.generations.length; ++i) {
            this.generations[i] = new ArrayList<SSTableReader>();
            this.lastCompactedKeys[i] = cfs.getPartitioner().getMinimumToken().minKeyBound();
        }
        this.compactionCounter = new int[MAX_LEVEL_COUNT];
    }

    public static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize, int fanoutSize, List<SSTableReader> sstables) {
        return LeveledManifest.create(cfs, maxSSTableSize, fanoutSize, sstables, new SizeTieredCompactionStrategyOptions());
    }

    public static LeveledManifest create(ColumnFamilyStore cfs, int maxSSTableSize, int fanoutSize, Iterable<SSTableReader> sstables, SizeTieredCompactionStrategyOptions options) {
        LeveledManifest manifest = new LeveledManifest(cfs, maxSSTableSize, fanoutSize, options);
        for (SSTableReader ssTableReader : sstables) {
            manifest.add(ssTableReader);
        }
        for (int i = 1; i < manifest.getAllLevelSize().length; ++i) {
            manifest.repairOverlappingSSTables(i);
        }
        manifest.calculateLastCompactedKeys();
        return manifest;
    }

    public void calculateLastCompactedKeys() {
        for (int i = 0; i < this.generations.length - 1; ++i) {
            if (this.generations[i + 1].isEmpty()) continue;
            SSTableReader sstableWithMaxModificationTime = null;
            long maxModificationTime = Long.MIN_VALUE;
            for (SSTableReader ssTableReader : this.generations[i + 1]) {
                long modificationTime = ssTableReader.getCreationTimeFor(Component.DATA);
                if (modificationTime < maxModificationTime) continue;
                sstableWithMaxModificationTime = ssTableReader;
                maxModificationTime = modificationTime;
            }
            this.lastCompactedKeys[i] = sstableWithMaxModificationTime.last;
        }
    }

    public synchronized void add(SSTableReader reader) {
        int level = reader.getSSTableLevel();
        assert (level < this.generations.length) : "Invalid level " + level + " out of " + (this.generations.length - 1);
        this.logDistribution();
        if (this.canAddSSTable(reader)) {
            logger.trace("Adding {} to L{}", (Object)reader, (Object)level);
            this.generations[level].add(reader);
        } else {
            try {
                reader.descriptor.getMetadataSerializer().mutateLevel(reader.descriptor, 0);
                reader.reloadSSTableMetadata();
            }
            catch (IOException e) {
                logger.error("Could not change sstable level - adding it at level 0 anyway, we will find it at restart.", (Throwable)e);
            }
            this.generations[0].add(reader);
        }
    }

    public synchronized void replace(Collection<SSTableReader> removed, Collection<SSTableReader> added) {
        assert (!removed.isEmpty());
        this.logDistribution();
        if (logger.isTraceEnabled()) {
            logger.trace("Replacing [{}]", (Object)this.toString(removed));
        }
        int minLevel = Integer.MAX_VALUE;
        for (SSTableReader sstable : removed) {
            int thisLevel = this.remove(sstable);
            minLevel = Math.min(minLevel, thisLevel);
        }
        if (added.isEmpty()) {
            return;
        }
        if (logger.isTraceEnabled()) {
            logger.trace("Adding [{}]", (Object)this.toString(added));
        }
        for (SSTableReader ssTableReader : added) {
            this.add(ssTableReader);
        }
        this.lastCompactedKeys[minLevel] = ((SSTableReader)SSTableReader.sstableOrdering.max(added)).last;
    }

    public synchronized void repairOverlappingSSTables(int level) {
        SSTableReader previous = null;
        Collections.sort(this.generations[level], SSTableReader.sstableComparator);
        ArrayList<SSTableReader> outOfOrderSSTables = new ArrayList<SSTableReader>();
        for (SSTableReader current : this.generations[level]) {
            if (previous != null && current.first.compareTo(previous.last) <= 0) {
                logger.warn("At level {}, {} [{}, {}] overlaps {} [{}, {}].  This could be caused by a bug in Cassandra 1.1.0 .. 1.1.3 or due to the fact that you have dropped sstables from another node into the data directory. Sending back to L0.  If you didn't drop in sstables, and have not yet run scrub, you should do so since you may also have rows out-of-order within an sstable", new Object[]{level, previous, previous.first, previous.last, current, current.first, current.last});
                outOfOrderSSTables.add(current);
                continue;
            }
            previous = current;
        }
        if (!outOfOrderSSTables.isEmpty()) {
            for (SSTableReader sstable : outOfOrderSSTables) {
                this.sendBackToL0(sstable);
            }
        }
    }

    private boolean canAddSSTable(SSTableReader sstable) {
        int level = sstable.getSSTableLevel();
        if (level == 0) {
            return true;
        }
        ArrayList<SSTableReader> copyLevel = new ArrayList<SSTableReader>(this.generations[level]);
        copyLevel.add(sstable);
        Collections.sort(copyLevel, SSTableReader.sstableComparator);
        SSTableReader previous = null;
        for (SSTableReader current : copyLevel) {
            if (previous != null && current.first.compareTo(previous.last) <= 0) {
                return false;
            }
            previous = current;
        }
        return true;
    }

    private synchronized void sendBackToL0(SSTableReader sstable) {
        this.remove(sstable);
        try {
            sstable.descriptor.getMetadataSerializer().mutateLevel(sstable.descriptor, 0);
            sstable.reloadSSTableMetadata();
            this.add(sstable);
        }
        catch (IOException e) {
            throw new RuntimeException("Could not reload sstable meta data", e);
        }
    }

    private String toString(Collection<SSTableReader> sstables) {
        StringBuilder builder = new StringBuilder();
        for (SSTableReader sstable : sstables) {
            builder.append(sstable.descriptor.cfname).append('-').append(sstable.descriptor.generation).append("(L").append(sstable.getSSTableLevel()).append("), ");
        }
        return builder.toString();
    }

    public long maxBytesForLevel(int level, long maxSSTableSizeInBytes) {
        return LeveledManifest.maxBytesForLevel(level, this.levelFanoutSize, maxSSTableSizeInBytes);
    }

    public static long maxBytesForLevel(int level, int levelFanoutSize, long maxSSTableSizeInBytes) {
        if (level == 0) {
            return 4L * maxSSTableSizeInBytes;
        }
        double bytes = Math.pow(levelFanoutSize, level) * (double)maxSSTableSizeInBytes;
        if (bytes > 9.223372036854776E18) {
            throw new RuntimeException("At most 9223372036854775807 bytes may be in a compaction level; your maxSSTableSize must be absurdly high to compute " + bytes);
        }
        return (long)bytes;
    }

    public synchronized CompactionCandidate getCompactionCandidates() {
        if (StorageService.instance.isBootstrapMode()) {
            List<SSTableReader> mostInteresting = this.getSSTablesForSTCS(this.getLevel(0));
            if (!mostInteresting.isEmpty()) {
                logger.info("Bootstrapping - doing STCS in L0");
                return new CompactionCandidate(mostInteresting, 0, Long.MAX_VALUE);
            }
            return null;
        }
        for (int i = this.generations.length - 1; i > 0; --i) {
            List<SSTableReader> sstables = this.getLevel(i);
            if (sstables.isEmpty()) continue;
            HashSet sstablesInLevel = Sets.newHashSet(sstables);
            Sets.SetView remaining = Sets.difference((Set)sstablesInLevel, this.cfs.getTracker().getCompacting());
            double score = (double)SSTableReader.getTotalBytes((Iterable<SSTableReader>)remaining) / (double)this.maxBytesForLevel(i, this.maxSSTableSizeInBytes);
            logger.trace("Compaction score for level {} is {}", (Object)i, (Object)score);
            if (!(score > 1.001)) continue;
            CompactionCandidate l0Compaction = this.getSTCSInL0CompactionCandidate();
            if (l0Compaction != null) {
                return l0Compaction;
            }
            Collection<SSTableReader> candidates = this.getCandidatesFor(i);
            if (!candidates.isEmpty()) {
                int nextLevel = this.getNextLevel(candidates);
                candidates = this.getOverlappingStarvedSSTables(nextLevel, candidates);
                if (logger.isTraceEnabled()) {
                    logger.trace("Compaction candidates for L{} are {}", (Object)i, (Object)this.toString(candidates));
                }
                return new CompactionCandidate(candidates, nextLevel, this.cfs.getCompactionStrategyManager().getMaxSSTableBytes());
            }
            logger.trace("No compaction candidates for L{}", (Object)i);
        }
        if (this.getLevel(0).isEmpty()) {
            return null;
        }
        Collection<SSTableReader> candidates = this.getCandidatesFor(0);
        if (candidates.isEmpty()) {
            return this.getSTCSInL0CompactionCandidate();
        }
        return new CompactionCandidate(candidates, this.getNextLevel(candidates), this.maxSSTableSizeInBytes);
    }

    private CompactionCandidate getSTCSInL0CompactionCandidate() {
        List<SSTableReader> mostInteresting;
        if (!DatabaseDescriptor.getDisableSTCSInL0() && this.getLevel(0).size() > 32 && !(mostInteresting = this.getSSTablesForSTCS(this.getLevel(0))).isEmpty()) {
            logger.debug("L0 is too far behind, performing size-tiering there first");
            return new CompactionCandidate(mostInteresting, 0, Long.MAX_VALUE);
        }
        return null;
    }

    private List<SSTableReader> getSSTablesForSTCS(Collection<SSTableReader> sstables) {
        Iterable<SSTableReader> candidates = this.cfs.getTracker().getUncompacting(sstables);
        List pairs = SizeTieredCompactionStrategy.createSSTableAndLengthPairs(AbstractCompactionStrategy.filterSuspectSSTables(candidates));
        List<List<SSTableReader>> buckets = SizeTieredCompactionStrategy.getBuckets(pairs, this.options.bucketHigh, this.options.bucketLow, this.options.minSSTableSize);
        return SizeTieredCompactionStrategy.mostInterestingBucket(buckets, 4, 32);
    }

    private Collection<SSTableReader> getOverlappingStarvedSSTables(int targetLevel, Collection<SSTableReader> candidates) {
        HashSet<SSTableReader> withStarvedCandidate = new HashSet<SSTableReader>(candidates);
        int i = this.generations.length - 1;
        while (i > 0) {
            int n = i--;
            this.compactionCounter[n] = this.compactionCounter[n] + 1;
        }
        this.compactionCounter[targetLevel] = 0;
        if (logger.isTraceEnabled()) {
            for (int j = 0; j < this.compactionCounter.length; ++j) {
                logger.trace("CompactionCounter: {}: {}", (Object)j, (Object)this.compactionCounter[j]);
            }
        }
        for (i = this.generations.length - 1; i > 0; --i) {
            if (this.getLevelSize(i) <= 0) continue;
            if (this.compactionCounter[i] > 25) {
                DecoratedKey max = null;
                Object min = null;
                for (SSTableReader candidate : candidates) {
                    if (min == null || candidate.first.compareTo((PartitionPosition)min) < 0) {
                        min = candidate.first;
                    }
                    if (max != null && candidate.last.compareTo(max) <= 0) continue;
                    max = candidate.last;
                }
                if (min == null || max == null || min.equals(max)) {
                    return candidates;
                }
                Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
                Range<DecoratedKey> boundaries = new Range<DecoratedKey>((DecoratedKey)min, max);
                for (SSTableReader sstable : this.getLevel(i)) {
                    Range<DecoratedKey> r = new Range<DecoratedKey>(sstable.first, sstable.last);
                    if (!boundaries.contains((DecoratedKey)((Object)r)) || compacting.contains(sstable)) continue;
                    logger.info("Adding high-level (L{}) {} to candidates", (Object)sstable.getSSTableLevel(), (Object)sstable);
                    withStarvedCandidate.add(sstable);
                    return withStarvedCandidate;
                }
            }
            return candidates;
        }
        return candidates;
    }

    public synchronized int getLevelSize(int i) {
        if (i >= this.generations.length) {
            throw new ArrayIndexOutOfBoundsException("Maximum valid generation is " + (this.generations.length - 1));
        }
        return this.getLevel(i).size();
    }

    public synchronized int[] getAllLevelSize() {
        int[] counts = new int[this.generations.length];
        for (int i = 0; i < counts.length; ++i) {
            counts[i] = this.getLevel(i).size();
        }
        return counts;
    }

    private void logDistribution() {
        if (logger.isTraceEnabled()) {
            for (int i = 0; i < this.generations.length; ++i) {
                if (this.getLevel(i).isEmpty()) continue;
                logger.trace("L{} contains {} SSTables ({}) in {}", new Object[]{i, this.getLevel(i).size(), FBUtilities.prettyPrintMemory(SSTableReader.getTotalBytes(this.getLevel(i))), this});
            }
        }
    }

    @VisibleForTesting
    public synchronized int remove(SSTableReader reader) {
        int level = reader.getSSTableLevel();
        assert (level >= 0) : reader + " not present in manifest: " + level;
        this.generations[level].remove(reader);
        return level;
    }

    private static Set<SSTableReader> overlapping(Collection<SSTableReader> candidates, Iterable<SSTableReader> others) {
        assert (!candidates.isEmpty());
        Iterator<SSTableReader> iter = candidates.iterator();
        SSTableReader sstable = iter.next();
        Token first = sstable.first.getToken();
        Token last = sstable.last.getToken();
        while (iter.hasNext()) {
            sstable = iter.next();
            first = first.compareTo(sstable.first.getToken()) <= 0 ? first : sstable.first.getToken();
            last = last.compareTo(sstable.last.getToken()) >= 0 ? last : sstable.last.getToken();
        }
        return LeveledManifest.overlapping(first, last, others);
    }

    private static Set<SSTableReader> overlappingWithBounds(SSTableReader sstable, Map<SSTableReader, Bounds<Token>> others) {
        return LeveledManifest.overlappingWithBounds(sstable.first.getToken(), sstable.last.getToken(), others);
    }

    @VisibleForTesting
    static Set<SSTableReader> overlapping(Token start, Token end, Iterable<SSTableReader> sstables) {
        return LeveledManifest.overlappingWithBounds(start, end, LeveledManifest.genBounds(sstables));
    }

    private static Set<SSTableReader> overlappingWithBounds(Token start, Token end, Map<SSTableReader, Bounds<Token>> sstables) {
        assert (start.compareTo(end) <= 0);
        HashSet<SSTableReader> overlapped = new HashSet<SSTableReader>();
        Bounds<Token> promotedBounds = new Bounds<Token>(start, end);
        for (Map.Entry<SSTableReader, Bounds<Token>> pair : sstables.entrySet()) {
            if (!pair.getValue().intersects(promotedBounds)) continue;
            overlapped.add(pair.getKey());
        }
        return overlapped;
    }

    private static Map<SSTableReader, Bounds<Token>> genBounds(Iterable<SSTableReader> ssTableReaders) {
        HashMap<SSTableReader, Bounds<Token>> boundsMap = new HashMap<SSTableReader, Bounds<Token>>();
        for (SSTableReader sstable : ssTableReaders) {
            boundsMap.put(sstable, new Bounds<Token>(sstable.first.getToken(), sstable.last.getToken()));
        }
        return boundsMap;
    }

    private Collection<SSTableReader> getCandidatesFor(int level) {
        assert (!this.getLevel(level).isEmpty());
        logger.trace("Choosing candidates for L{}", (Object)level);
        Set<SSTableReader> compacting = this.cfs.getTracker().getCompacting();
        if (level == 0) {
            Set<SSTableReader> compactingL0 = this.getCompacting(0);
            DecoratedKey lastCompactingKey = null;
            DecoratedKey firstCompactingKey = null;
            for (SSTableReader candidate : compactingL0) {
                if (firstCompactingKey == null || candidate.first.compareTo(firstCompactingKey) < 0) {
                    firstCompactingKey = candidate.first;
                }
                if (lastCompactingKey != null && candidate.last.compareTo(lastCompactingKey) <= 0) continue;
                lastCompactingKey = candidate.last;
            }
            Object candidates = new HashSet();
            Map<SSTableReader, Bounds<Token>> remaining = LeveledManifest.genBounds(Iterables.filter(this.getLevel(0), (Predicate)Predicates.not(suspectP)));
            for (SSTableReader sstable : this.ageSortedSSTables(remaining.keySet())) {
                Sets.SetView overlappedL0;
                if (candidates.contains(sstable) || !Sets.intersection((Set)(overlappedL0 = Sets.union(Collections.singleton(sstable), LeveledManifest.overlappingWithBounds(sstable, remaining))), compactingL0).isEmpty()) continue;
                for (SSTableReader newCandidate : overlappedL0) {
                    if (firstCompactingKey == null || lastCompactingKey == null || LeveledManifest.overlapping(firstCompactingKey.getToken(), lastCompactingKey.getToken(), Arrays.asList(newCandidate)).size() == 0) {
                        candidates.add(newCandidate);
                    }
                    remaining.remove(newCandidate);
                }
                if (candidates.size() <= 32) continue;
                candidates = new HashSet<SSTableReader>(this.ageSortedSSTables((Collection<SSTableReader>)candidates).subList(0, 32));
                break;
            }
            if (SSTableReader.getTotalBytes(candidates) > this.maxSSTableSizeInBytes) {
                Set<SSTableReader> l1overlapping = LeveledManifest.overlapping(candidates, this.getLevel(1));
                if (Sets.intersection(l1overlapping, compacting).size() > 0) {
                    return Collections.emptyList();
                }
                if (!LeveledManifest.overlapping(candidates, compactingL0).isEmpty()) {
                    return Collections.emptyList();
                }
                candidates = Sets.union(candidates, l1overlapping);
            }
            if (candidates.size() < 2) {
                return Collections.emptyList();
            }
            return candidates;
        }
        Collections.sort(this.getLevel(level), SSTableReader.sstableComparator);
        int start = 0;
        for (int i = 0; i < this.getLevel(level).size(); ++i) {
            SSTableReader sstable = this.getLevel(level).get(i);
            if (sstable.first.compareTo(this.lastCompactedKeys[level]) <= 0) continue;
            start = i;
            break;
        }
        Map<SSTableReader, Bounds<Token>> sstablesNextLevel = LeveledManifest.genBounds(this.getLevel(level + 1));
        for (int i = 0; i < this.getLevel(level).size(); ++i) {
            SSTableReader sstable = this.getLevel(level).get((start + i) % this.getLevel(level).size());
            Sets.SetView candidates = Sets.union(Collections.singleton(sstable), LeveledManifest.overlappingWithBounds(sstable, sstablesNextLevel));
            if (Iterables.any((Iterable)candidates, suspectP) || !Sets.intersection((Set)candidates, compacting).isEmpty()) continue;
            return candidates;
        }
        return Collections.emptyList();
    }

    private Set<SSTableReader> getCompacting(int level) {
        HashSet<SSTableReader> sstables = new HashSet<SSTableReader>();
        HashSet<SSTableReader> levelSSTables = new HashSet<SSTableReader>(this.getLevel(level));
        for (SSTableReader sstable : this.cfs.getTracker().getCompacting()) {
            if (!levelSSTables.contains(sstable)) continue;
            sstables.add(sstable);
        }
        return sstables;
    }

    private List<SSTableReader> ageSortedSSTables(Collection<SSTableReader> candidates) {
        ArrayList<SSTableReader> ageSortedCandidates = new ArrayList<SSTableReader>(candidates);
        Collections.sort(ageSortedCandidates, SSTableReader.maxTimestampComparator);
        return ageSortedCandidates;
    }

    public synchronized Set<SSTableReader>[] getSStablesPerLevelSnapshot() {
        Set[] sstablesPerLevel = new Set[this.generations.length];
        for (int i = 0; i < this.generations.length; ++i) {
            sstablesPerLevel[i] = new HashSet<SSTableReader>(this.generations[i]);
        }
        return sstablesPerLevel;
    }

    public String toString() {
        return "Manifest@" + this.hashCode();
    }

    public int getLevelCount() {
        for (int i = this.generations.length - 1; i >= 0; --i) {
            if (this.getLevel(i).size() <= 0) continue;
            return i;
        }
        return 0;
    }

    public synchronized SortedSet<SSTableReader> getLevelSorted(int level, Comparator<SSTableReader> comparator) {
        return ImmutableSortedSet.copyOf(comparator, this.getLevel(level));
    }

    public List<SSTableReader> getLevel(int i) {
        return this.generations[i];
    }

    public synchronized int getEstimatedTasks() {
        long tasks = 0L;
        long[] estimated = new long[this.generations.length];
        for (int i = this.generations.length - 1; i >= 0; --i) {
            List<SSTableReader> sstables = this.getLevel(i);
            estimated[i] = (long)Math.ceil((double)Math.max(0L, SSTableReader.getTotalBytes(sstables) - (long)((double)this.maxBytesForLevel(i, this.maxSSTableSizeInBytes) * 1.001)) / (double)this.maxSSTableSizeInBytes);
            tasks += estimated[i];
        }
        logger.trace("Estimating {} compactions to do for {}.{}", new Object[]{Arrays.toString(estimated), this.cfs.keyspace.getName(), this.cfs.name});
        return Ints.checkedCast((long)tasks);
    }

    public int getNextLevel(Collection<SSTableReader> sstables) {
        int newLevel;
        int maximumLevel = Integer.MIN_VALUE;
        int minimumLevel = Integer.MAX_VALUE;
        for (SSTableReader sstable : sstables) {
            maximumLevel = Math.max(sstable.getSSTableLevel(), maximumLevel);
            minimumLevel = Math.min(sstable.getSSTableLevel(), minimumLevel);
        }
        if (minimumLevel == 0 && minimumLevel == maximumLevel && SSTableReader.getTotalBytes(sstables) < this.maxSSTableSizeInBytes) {
            newLevel = 0;
        } else {
            int n = newLevel = minimumLevel == maximumLevel ? maximumLevel + 1 : maximumLevel;
            assert (newLevel > 0);
        }
        return newLevel;
    }

    public Iterable<SSTableReader> getAllSSTables() {
        HashSet<SSTableReader> sstables = new HashSet<SSTableReader>();
        for (List<SSTableReader> generation : this.generations) {
            sstables.addAll(generation);
        }
        return sstables;
    }

    public static class CompactionCandidate {
        public final Collection<SSTableReader> sstables;
        public final int level;
        public final long maxSSTableBytes;

        public CompactionCandidate(Collection<SSTableReader> sstables, int level, long maxSSTableBytes) {
            this.sstables = sstables;
            this.level = level;
            this.maxSSTableBytes = maxSSTableBytes;
        }
    }
}

